package reminder

import (
	"context"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/libs/go/xlock"
	"a.yandex-team.ru/security/skotty/service/internal/db"
	"a.yandex-team.ru/security/skotty/service/internal/mailer"
	"a.yandex-team.ru/security/skotty/service/internal/models"
	"a.yandex-team.ru/yt/go/yterrors"
)

const lockDelay = 10 * time.Minute

var _ Reminder = (*MailReminder)(nil)

type MailReminder struct {
	db         *db.DB
	lock       xlock.Locker
	mailer     *mailer.Mailer
	log        log.Logger
	remainDays []int
	ctx        context.Context
	cancelCtx  context.CancelFunc
	closed     chan struct{}
}

func NewMailReminder(opts ...Option) (*MailReminder, error) {
	ctx, cancelCtx := context.WithCancel(context.Background())

	r := &MailReminder{
		log:       &nop.Logger{},
		lock:      &xlock.NopLocker{},
		ctx:       ctx,
		cancelCtx: cancelCtx,
		closed:    make(chan struct{}),
		remainDays: []int{
			0, 3, 7,
		},
	}

	for _, opt := range opts {
		if err := opt(r); err != nil {
			return nil, err
		}
	}

	if r.db == nil {
		return nil, errors.New("reminder can't work w/o db, please pass WithDB option")
	}

	if r.mailer == nil {
		return nil, errors.New("revoker can't work w/o mailer, please pass WithMailer option")
	}

	go r.loop()
	return r, nil
}

func (r *MailReminder) loop() {
	defer close(r.closed)

	waitAndProcess := func(ctx context.Context) error {
		t := time.NewTimer(nextTick())
		defer t.Stop()

		select {
		case <-ctx.Done():
			return nil
		case <-t.C:
			return r.process()
		}
	}

	for {
		tx, err := r.lock.Lock(r.ctx)
		if err != nil {
			if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
				r.log.Info("conflict lock", log.String("error", err.Error()))
				time.Sleep(lockDelay)
				continue
			}

			r.log.Error("unable to acquire lock", log.Error(err))
			continue
		}

		err = waitAndProcess(tx.Context())
		_ = tx.Unlock()
		if err != nil {
			r.log.Error("process failed", log.Error(err))
		}

		if r.ctx.Err() != nil {
			return
		}
	}
}

func (r *MailReminder) process() error {
	now := time.Now()
	for _, remain := range r.remainDays {
		tokens, err := r.db.LookupExpiresTokens(r.ctx, now.Add(time.Duration(remain)*24*time.Hour))
		if err != nil {
			return fmt.Errorf("unable to list expired tokens: %w", err)
		}

		for _, token := range tokens {
			tfID := models.TFID(token.User, token.ID, token.EnrollID)
			r.log.Info("notify user about expires token",
				log.String("tfid", tfID),
				log.Int("days_remain", remain),
				log.Int64("expires_at", token.ExpiresAt))

			r.mailer.TokenExpires(r.ctx, token, remain)
		}
	}

	return nil
}

func (r *MailReminder) Shutdown(ctx context.Context) {
	r.cancelCtx()

	select {
	case <-ctx.Done():
	case <-r.closed:
	}
}

func nextTick() time.Duration {
	now := time.Now()
	n := time.Date(now.Year(), now.Month(), now.Day(), 12, 0, 0, 0, now.Location())
	if now.After(n) {
		n = n.Add(24 * time.Hour)
	}
	return n.Sub(now)
}
