package anysign

import (
	"fmt"
	"sync"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/signer"
	"a.yandex-team.ru/security/skotty/service/internal/signer/memsign"
	"a.yandex-team.ru/security/skotty/service/internal/signer/softsign"
	"a.yandex-team.ru/security/skotty/service/internal/signer/yubisign"
)

type AnySigner struct {
	signers map[skotty.TokenType]signer.Signer
	ca      *signer.CAStorage
	mu      sync.RWMutex
}

func NewSigner(caStorage *signer.CAStorage) (*AnySigner, error) {
	return &AnySigner{
		signers: make(map[skotty.TokenType]signer.Signer),
		ca:      caStorage,
	}, nil
}

func (s *AnySigner) Signer(tokenType skotty.TokenType) (signer.Signer, error) {
	s.mu.RLock()
	ss, ok := s.signers[tokenType]
	s.mu.RUnlock()
	if ok {
		return ss, nil
	}

	var err error
	switch tokenType {
	case skotty.TokenTypeYubikey:
		ss, err = yubisign.NewSigner(s.ca)
	case skotty.TokenTypeSoft:
		ss, err = softsign.NewSigner(s.ca)
	case skotty.TokenTypeMemory, skotty.TokenTypeRoboMemory:
		ss, err = memsign.NewSigner(s.ca)
	default:
		err = fmt.Errorf("unsupported token type: %s", tokenType)
	}

	if err != nil {
		return nil, err
	}

	s.mu.Lock()
	s.signers[tokenType] = ss
	s.mu.Unlock()
	return ss, nil
}
