package signer

import (
	"bytes"
	"encoding/json"
	"fmt"
	"os"
	"reflect"
	"time"

	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/security/skotty/libs/certutil"
)

type CertificateInfo struct {
	Name      string    `json:"name"`
	CreatedAt time.Time `json:"created_at"`
	Private   []byte    `json:"private,omitempty"`
	Public    []byte    `json:"public"`
	SSHPublic []byte    `json:"ssh_public"`
}

func parseCertificateInfo(filename string) (*certificate, error) {
	if filename == "" {
		return nil, nil
	}

	rawBytes, err := os.ReadFile(filename)
	if err != nil {
		return nil, fmt.Errorf("read cert info file: %w", err)
	}

	var ci CertificateInfo
	err = json.Unmarshal(rawBytes, &ci)
	if err != nil {
		return nil, fmt.Errorf("invald ca info: %w", err)
	}

	pubKey, err := certutil.PemToCert(ci.Public)
	if err != nil {
		return nil, fmt.Errorf("failed to parse pub key: %w", err)
	}

	authorizedKey, _, _, _, err := ssh.ParseAuthorizedKey(ci.SSHPublic)
	if err != nil {
		return nil, fmt.Errorf("failed to parse SSH pub key: %w", err)
	}

	out := certificate{
		Public:         pubKey,
		PublicBytes:    bytes.TrimSpace(ci.Public),
		SSHPublicBytes: bytes.TrimSpace(ci.SSHPublic),
		SSHFingerprint: ssh.FingerprintSHA256(authorizedKey),
	}

	if len(ci.Private) == 0 {
		return &out, nil
	}

	out.Private, err = certutil.PemToECPriv(ci.Private)
	if err != nil {
		return nil, fmt.Errorf("failed to parse priv key: %w", err)
	}

	if !reflect.DeepEqual(&out.Private.PublicKey, pubKey.PublicKey) {
		return nil, fmt.Errorf("x509 pub key doesn't match priv key: %v != %v", &out.Private.PublicKey, pubKey.PublicKey)
	}

	out.SSHSigner, err = ssh.NewSignerFromKey(out.Private)
	if err != nil {
		return nil, err
	}

	if !reflect.DeepEqual(out.SSHSigner.PublicKey(), authorizedKey) {
		return nil, fmt.Errorf("ssh pub key doesn't match priv key: %v != %v", out.SSHSigner.PublicKey(), authorizedKey)
	}

	return &out, nil
}
