package memsign

import (
	"crypto/x509"
	"fmt"
	"time"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/signer"
	"a.yandex-team.ru/security/skotty/skotty/pkg/softattest"
)

var _ signer.Signer = (*SoftSigner)(nil)

const certTTL = 86400 * time.Second

type SoftSigner struct {
	ca *signer.CAStorage
}

func NewSigner(ca *signer.CAStorage) (*SoftSigner, error) {
	return &SoftSigner{
		ca: ca,
	}, nil
}

func (s *SoftSigner) CertLifetime() time.Duration {
	return certTTL
}

func (s *SoftSigner) AttestSlot(certType skotty.CertType, attestationCert, slotCert *x509.Certificate) (*signer.Attestation, error) {
	switch certType {
	case skotty.CertTypeSecure, skotty.CertTypeInsecure, skotty.CertTypeSudo:
	default:
		return nil, fmt.Errorf("unsupported cert type: %s", certType)
	}

	attest, err := softattest.Verify(attestationCert, slotCert)
	if err != nil {
		return nil, fmt.Errorf("attestation failed: %w", err)
	}

	return &signer.Attestation{
		Serial: attest.Serial,
	}, nil
}

func (s *SoftSigner) IssueCertificate(certType skotty.CertType, csr *signer.CertificateRequest) (*signer.SSHCert, error) {
	return s.ca.Sign(certType, csr)
}
