package yubisign

import (
	"crypto/x509"
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/signer"
	"a.yandex-team.ru/security/skotty/service/internal/signer/yubisign/yubikey"
)

var _ signer.Signer = (*YubiSigner)(nil)

const (
	certTTL = 90 * 86400 * time.Second
)

var certMinTouchPolicy = map[skotty.CertType]yubikey.TouchPolicy{
	skotty.CertTypeSecure:   yubikey.TouchPolicyCached,
	skotty.CertTypeInsecure: yubikey.TouchPolicyNever,
	skotty.CertTypeSudo:     yubikey.TouchPolicyAlways,
	skotty.CertTypeLegacy:   yubikey.TouchPolicyNever,
	skotty.CertTypeRenew:    yubikey.TouchPolicyNever,
}

type YubiSigner struct {
	ca *signer.CAStorage
}

func NewSigner(ca *signer.CAStorage) (*YubiSigner, error) {
	return &YubiSigner{
		ca: ca,
	}, nil
}

func (s *YubiSigner) CertLifetime() time.Duration {
	return certTTL
}

func (s *YubiSigner) AttestSlot(certType skotty.CertType, attestationCert, slotCert *x509.Certificate) (*signer.Attestation, error) {
	attest, err := yubikey.Verify(attestationCert, slotCert)
	if err != nil {
		return nil, fmt.Errorf("attestation failed: %w", err)
	}

	if attest.PINPolicy < yubikey.PINPolicyOnce {
		return nil, fmt.Errorf("invalid PIN policy for cert type %q: %d < %d", certType, attest.TouchPolicy, yubikey.PINPolicyOnce)
	}

	minTouchPolicy, ok := certMinTouchPolicy[certType]
	if !ok {
		return nil, fmt.Errorf("unsupported cert type: %s", certType)
	}

	if attest.TouchPolicy < minTouchPolicy {
		return nil, fmt.Errorf("invalid touch policy for cert type %q: %d < %d", certType, attest.TouchPolicy, minTouchPolicy)
	}

	return &signer.Attestation{
		Serial: strconv.Itoa(int(attest.Serial)),
	}, nil
}

func (s *YubiSigner) IssueCertificate(certType skotty.CertType, csr *signer.CertificateRequest) (*signer.SSHCert, error) {
	return s.ca.Sign(certType, csr)
}
