package staff

import (
	"context"
	"crypto/tls"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

const (
	DefaultUpstream = "https://staff.yandex-team.ru"

	ProductionTVMID tvm.ClientID = 2000054
	TestingTVMID    tvm.ClientID = 2000053
)

type Client struct {
	httpc *resty.Client
	tvmc  tvm.Client
	tvmID tvm.ClientID
}

func NewClient(tvmc tvm.Client, opts ...Option) (*Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("can't create ca pool: %w", err)
	}

	httpc := resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetJSONEscapeHTML(false).
		SetHeader("Content-Type", "application/json").
		SetBaseURL(DefaultUpstream).
		SetTimeout(1 * time.Second).
		SetRetryCount(3).
		SetRetryWaitTime(100 * time.Millisecond).
		SetRetryMaxWaitTime(3 * time.Second).
		AddRetryCondition(func(rsp *resty.Response, err error) bool {
			return err != nil || rsp.StatusCode() == http.StatusInternalServerError
		})

	c := &Client{
		httpc: httpc,
		tvmc:  tvmc,
		tvmID: ProductionTVMID,
	}
	for _, o := range opts {
		o(c)
	}

	return c, nil
}

func (c *Client) AddSSHKey(ctx context.Context, userTicket string, keys ...SSHKey) error {
	if len(keys) == 0 {
		return nil
	}

	serviceTicket, err := c.tvmc.GetServiceTicketForID(ctx, c.tvmID)
	if err != nil {
		return fmt.Errorf("unable to get staff TVM ticket: %w", err)
	}

	var result staffResponse
	var errResult staffError
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetResult(&result).
		SetError(&errResult).
		SetBody(keys).
		SetHeader("X-Ya-Service-Ticket", serviceTicket).
		SetHeader("X-Ya-User-Ticket", userTicket).
		ForceContentType("application/json").
		Post("/keys/skotty-upload-ssh-keys")

	if err != nil {
		return fmt.Errorf("request failed: %w", err)
	}

	if !result.Success {
		return fmt.Errorf("remote error: %s", strings.Join(errResult.Errors, "."))
	}

	if !rsp.IsSuccess() {
		return fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return nil
}

func (c *Client) RemoveSSHKey(ctx context.Context, login string, fingerprints ...string) error {
	if len(fingerprints) == 0 {
		return nil
	}

	serviceTicket, err := c.tvmc.GetServiceTicketForID(ctx, c.tvmID)
	if err != nil {
		return fmt.Errorf("unable to get staff TVM ticket: %w", err)
	}

	var result staffResponse
	var errResult staffError
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetResult(&result).
		SetError(&errResult).
		SetHeader("X-Ya-Service-Ticket", serviceTicket).
		SetBody(struct {
			Login        string   `json:"login"`
			Fingerprints []string `json:"fingerprints"`
		}{
			Login:        login,
			Fingerprints: fingerprints,
		}).
		ForceContentType("application/json").
		Post("/keys/skotty-revoke-ssh-keys")

	if err != nil {
		return fmt.Errorf("request failed: %w", err)
	}

	if !result.Success {
		return fmt.Errorf("remote error: %s", strings.Join(errResult.Errors, "."))
	}

	if !rsp.IsSuccess() {
		return fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return nil
}
