package storage

import (
	"context"
	"encoding/base64"
	"fmt"
	"io"
	"net/http"
	"net/http/httputil"
	"path"

	"cuelang.org/go/pkg/strings"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/library/go/yandex/awstvm"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

type Storage struct {
	endpoint      string
	bucket        string
	accessKeyID   string
	tvmID         tvm.ClientID
	s3Client      *s3.S3
	uploader      *s3manager.Uploader
	downloader    *s3manager.Downloader
	downloadProxy *httputil.ReverseProxy
}

func NewStorage(tvmc tvm.Client, opts ...Option) (*Storage, error) {
	storage := &Storage{
		endpoint: "s3.mds.yandex.net",
		bucket:   "skotty",
	}

	for _, opt := range opts {
		opt(storage)
	}

	if err := storage.initS3Client(tvmc); err != nil {
		return nil, err
	}

	if err := storage.initBucket(); err != nil {
		return nil, err
	}

	storage.initUploader()
	storage.initDownloader()
	storage.initDownloadProxy()

	return storage, nil
}

func (s *Storage) initDownloadProxy() {
	host := fmt.Sprintf("%s.%s", s.bucket, s.endpoint)

	director := func(req *http.Request) {
		req.URL.Scheme = "http"
		req.URL.Host = host
	}

	s.downloadProxy = &httputil.ReverseProxy{
		Director: director,
	}
}

func (s *Storage) initBucket() error {
	if _, err := s.s3Client.HeadBucket(&s3.HeadBucketInput{Bucket: aws.String(s.bucket)}); err != nil {
		_, err := s.s3Client.CreateBucket(&s3.CreateBucketInput{Bucket: aws.String(s.bucket)})
		if err != nil {
			return err
		}
	}
	return nil
}

func (s *Storage) initS3Client(tvmc tvm.Client) error {
	creds, err := awstvm.NewS3Credentials(tvmc,
		awstvm.WithTvmClientID(s.tvmID),
		awstvm.WithAccessKeyID(s.accessKeyID),
	)
	if err != nil {
		return fmt.Errorf("failed to create s3 credentials: %w", err)
	}

	cfg := aws.NewConfig().
		WithRegion("dummy").
		WithEndpoint(s.endpoint).
		WithCredentials(creds)

	s3Session, err := session.NewSession()
	if err != nil {
		return fmt.Errorf("failed to create session: %w", err)
	}

	s.s3Client = s3.New(s3Session, cfg)
	return nil
}

func (s *Storage) initUploader() {
	s.uploader = s3manager.NewUploaderWithClient(s.s3Client, func(u *s3manager.Uploader) {
		u.PartSize = 32 * 1024 * 1024
	})
}

func (s *Storage) initDownloader() {
	s.downloader = s3manager.NewDownloaderWithClient(s.s3Client, func(d *s3manager.Downloader) {
		d.Concurrency = 1
		d.PartSize = 32 * 1024 * 1024
	})
}

func (s *Storage) Exists(ctx context.Context, path string) (bool, error) {
	_, err := s.s3Client.HeadObjectWithContext(ctx, &s3.HeadObjectInput{
		Bucket: aws.String(s.bucket),
		Key:    aws.String(path),
	})
	if err != nil {
		if isAwsError(err, s3.ErrCodeNoSuchKey) || isAwsError(err, "NotFound") {
			return false, nil
		}

		return false, err
	}

	return true, nil
}

type UploadReq struct {
	Path        string
	ContentType string
	MD5Sum      []byte
	Data        io.Reader
}

func (s *Storage) Upload(ctx context.Context, req *UploadReq) (string, error) {
	uploadReq := &s3manager.UploadInput{
		Bucket: aws.String(s.bucket),
		Key:    aws.String(req.Path),
		Body:   req.Data,
	}
	if req.ContentType != "" {
		uploadReq.ContentType = aws.String(req.ContentType)

	}
	if len(req.MD5Sum) > 0 {
		uploadReq.ContentMD5 = aws.String(base64.StdEncoding.EncodeToString(req.MD5Sum))
	}

	_, err := s.uploader.UploadWithContext(ctx, uploadReq)
	if err != nil {
		return "", err
	}

	return fmt.Sprintf("https://%s.%s/%s", s.bucket, s.endpoint, req.Path), nil
}

func (s *Storage) Download(ctx context.Context, path string, out io.Writer) error {
	_, err := s.downloader.DownloadWithContext(ctx, FakeWriterAt{out}, &s3.GetObjectInput{
		Bucket: aws.String(s.bucket),
		Key:    aws.String(path),
	})

	return err
}

func (s *Storage) List(ctx context.Context, path string) ([]string, error) {
	result, err := s.s3Client.ListObjectsWithContext(ctx, &s3.ListObjectsInput{
		Bucket:    aws.String(s.bucket),
		Prefix:    ptr.String(path),
		Delimiter: ptr.String("/"),
	})
	if err != nil {
		return nil, err
	}

	out := make([]string, len(result.CommonPrefixes))
	for i, prefix := range result.CommonPrefixes {
		out[i] = strings.Trim(strings.TrimPrefix(*prefix.Prefix, path), "/")
	}

	return out, err
}

func (s *Storage) Copy(ctx context.Context, from, to string) error {
	_, err := s.s3Client.CopyObjectWithContext(ctx, &s3.CopyObjectInput{
		Bucket:     aws.String(s.bucket),
		CopySource: aws.String(path.Join(s.bucket, from)),
		Key:        aws.String(to),
	})
	return err
}

func (s *Storage) Serve(r *http.Request, path string, w http.ResponseWriter) error {
	uri := fmt.Sprintf("http://%s.%s/%s", s.bucket, s.endpoint, path)
	req, err := http.NewRequestWithContext(r.Context(), "GET", uri, nil)
	if err != nil {
		return fmt.Errorf("failed to create proxy request: %w", err)
	}

	if ifNoneMatch := r.Header.Get("If-None-Match"); ifNoneMatch != "" {
		req.Header.Set("If-None-Match", ifNoneMatch)
	}

	s.downloadProxy.ServeHTTP(w, req)
	return nil
}

func isAwsError(err error, code string) bool {
	if err, ok := err.(awserr.Error); ok && err.Code() == code {
		return true
	}
	return false
}

type FakeWriterAt struct {
	w io.Writer
}

// WriteAt writes to the writer and ignores the offset
func (fw FakeWriterAt) WriteAt(p []byte, _ int64) (n int, err error) {
	return fw.w.Write(p)
}
