package webauth

import (
	"errors"
	"fmt"
	"time"

	"github.com/golang-jwt/jwt/v4"

	"a.yandex-team.ru/security/skotty/service/internal/models"
)

type StandardClaims struct {
	ExpiresAt int64           `json:"exp,omitempty"`
	ID        string          `json:"jti,omitempty"`
	IssuedAt  int64           `json:"iat,omitempty"`
	Issuer    string          `json:"iss,omitempty"`
	AuthKind  models.AuthKind `json:"ak,omitempty"`
}

type AuthClaims struct {
	StandardClaims
	User         string           `json:"user,omitempty"`
	TokenType    models.TokenType `json:"tt,omitempty"`
	TokenSerial  string           `json:"ts,omitempty"`
	TokenName    string           `json:"tn,omitempty"`
	EnrollmentID string           `json:"ei,omitempty"`
	Hostname     string           `json:"host,omitempty"`
}

type RenewClaims struct {
	StandardClaims
	User         string           `json:"user,omitempty"`
	TokenSerial  string           `json:"ts,omitempty"`
	TokenType    models.TokenType `json:"tt,omitempty"`
	EnrollmentID string           `json:"ei,omitempty"`
}

func (c StandardClaims) Valid() error {
	vErr := new(jwt.ValidationError)
	now := time.Now().Unix()

	if c.ID == "" {
		vErr.Inner = errors.New("no auth id")
		vErr.Errors |= jwt.ValidationErrorId
	}

	if c.Issuer != Issuer {
		vErr.Inner = fmt.Errorf("unexpected issuer: %s", c.Issuer)
		vErr.Errors |= jwt.ValidationErrorIssuer
	}

	if now < c.IssuedAt {
		vErr.Inner = fmt.Errorf("token used before issued: %d > %d", now, c.IssuedAt)
		vErr.Errors |= jwt.ValidationErrorIssuedAt
	}

	if now >= c.ExpiresAt {
		delta := time.Unix(now, 0).Sub(time.Unix(c.ExpiresAt, 0))
		vErr.Inner = fmt.Errorf("token is expired by %v", delta)
		vErr.Errors |= jwt.ValidationErrorExpired
	}

	if c.AuthKind == models.AuthKindNone {
		vErr.Inner = fmt.Errorf("invalud auth kind: %s", c.AuthKind)
		vErr.Errors |= jwt.ValidationErrorClaimsInvalid
	}

	if vErr.Errors == 0 {
		return nil
	}

	return vErr
}
