package yasms

import (
	"context"
	"crypto/tls"
	"encoding/json"
	"encoding/xml"
	"errors"
	"fmt"
	"net/http"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

const (
	DefaultUpstream = "https://sms.passport.yandex.ru"
	DefaultSender   = "skotty"
	TVMClientIDProd = 2000836
	TVMClientIDTest = 2000834
)

type Env struct {
	Upstream    string
	Sender      string
	TVMClientID tvm.ClientID
}

var _ Client = (*HTTPClient)(nil)

type HTTPClient struct {
	env   Env
	tvmc  tvm.Client
	httpc *resty.Client
}

func NewHTTPClient(opts ...Option) (*HTTPClient, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("can't create ca pool: %w", err)
	}

	httpc := resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetRetryCount(3).
		SetRetryWaitTime(100 * time.Millisecond).
		SetRetryMaxWaitTime(3 * time.Second).
		AddRetryCondition(func(rsp *resty.Response, err error) bool {
			return err != nil || rsp.StatusCode() == http.StatusInternalServerError
		})

	c := &HTTPClient{
		env: Env{
			Upstream:    DefaultUpstream,
			Sender:      DefaultSender,
			TVMClientID: TVMClientIDProd,
		},
		httpc: httpc,
	}
	for _, o := range opts {
		o(c)
	}

	if c.tvmc == nil {
		return nil, errors.New("no TVM client configured")
	}

	return c, nil
}

func (c *HTTPClient) Send(ctx context.Context, phone, template string, params map[string]string) (string, error) {
	ticket, err := c.tvmc.GetServiceTicketForID(ctx, c.env.TVMClientID)
	if err != nil {
		return "", fmt.Errorf("get service ticket: %w", err)
	}

	formParams, err := json.Marshal(params)
	if err != nil {
		return "", fmt.Errorf("unable to encode template params: %w", err)
	}

	var result struct {
		XMLName xml.Name `xml:"doc"`
		Message struct {
			ID string `xml:"id,attr"`
		} `xml:"message-sent"`
		Gates struct {
			IDs string `xml:"ids,attr"`
		} `xml:"gates"`
		Error     string `xml:"error"`
		ErrorCode string `xml:"errorcode"`
	}

	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetResult(&result).
		ForceContentType("application/xml").
		SetHeader("X-Ya-Service-Ticket", ticket).
		SetQueryParams(map[string]string{
			"phone":  phone,
			"text":   template,
			"sender": c.env.Sender,
		}).
		SetFormData(map[string]string{
			"text_template_params": string(formParams),
		}).
		Post(c.env.Upstream + "/sendsms")

	if err != nil {
		return "", fmt.Errorf("request failed: %w", err)
	}

	if result.Error != "" {
		return "", fmt.Errorf("remote error: [%s] %s", result.ErrorCode, result.Error)
	}

	if !rsp.IsSuccess() {
		return "", fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return fmt.Sprintf("%s@%s", result.Message.ID, result.Gates.IDs), nil
}
