package agent

import (
	"context"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"
)

type Agent interface {
	SSHAgent
	Name() string
	ReloadKeys()
	ListenAndServe(ctx context.Context) error
	Shutdown(ctx context.Context)
}

type SSHAgent interface {
	// List returns the identities known to the agent.
	List(ctx context.Context) ([]*agent.Key, error)

	// Sign has the agent sign the data using a protocol 2 key as defined
	// in [PROTOCOL.agent] section 2.6.2.
	Sign(ctx context.Context, key ssh.PublicKey, data []byte) (*ssh.Signature, error)

	// Add adds a private key to the agent.
	Add(ctx context.Context, key agent.AddedKey) error

	// Remove removes all identities with the given public key.
	Remove(ctx context.Context, key ssh.PublicKey) error

	// RemoveAll removes all identities.
	RemoveAll(ctx context.Context) error

	// Lock locks the agent. Sign and Remove will fail, and List will empty an empty list.
	Lock(ctx context.Context, passphrase []byte) error

	// Unlock undoes the effect of Lock
	Unlock(ctx context.Context, passphrase []byte) error

	// Signers returns signers for all the known keys.
	Signers(ctx context.Context) ([]ssh.Signer, error)

	// SignWithFlags signs like Sign, but allows for additional flags to be sent/received
	SignWithFlags(ctx context.Context, key ssh.PublicKey, data []byte, flags agent.SignatureFlags) (*ssh.Signature, error)

	// Extension processes a custom extension request. Standard-compliant agents are not
	// required to support any extensions, but this method allows agents to implement
	// vendor-specific methods or add experimental features. See [PROTOCOL.agent] section 4.7.
	// If agent extensions are unsupported entirely this method MUST return an
	// ErrExtensionUnsupported error. Similarly, if just the specific extensionType in
	// the request is unsupported by the agent then ErrExtensionUnsupported MUST be
	// returned.
	//
	// In the case of success, since [PROTOCOL.agent] section 4.7 specifies that the contents
	// of the response are unspecified (including the type of the message), the complete
	// response will be returned as a []byte slice, including the "type" byte of the message.
	Extension(ctx context.Context, extensionType string, contents []byte) ([]byte, error)
}
