package agent

import (
	"context"
	"fmt"
	"sync"

	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/confirm"
	"a.yandex-team.ru/security/skotty/skotty/internal/ui"
)

type AgentsGang struct {
	agents []Agent
	notify ui.Notifier
}

func NewAgentsGang(store *KeyHolder, opts ...Option) (*AgentsGang, error) {
	var sockets []config.Socket
	var logger log.Logger = &nop.Logger{}
	var notify ui.Notifier
	var confirmator confirm.Confirmator
	for _, opt := range opts {
		switch o := opt.(type) {
		case loggerOption:
			logger = o.logger
		case socketsOption:
			sockets = o.sockets
		case notifierOption:
			notify = o.notifier
		case confirmatorOption:
			confirmator = o.confirmator

		default:
			return nil, fmt.Errorf("unexpected agent gang option: %T", o)
		}
	}

	// first of all - create real agents
	i := 0
	agents := make([]Agent, 0, len(sockets))
	realAgents := make(map[string]int, len(sockets))
	for _, sock := range sockets {
		if sock.SameAs != "" {
			continue
		}

		agent, err := newRealAgent(RealAgentParams{
			store:   store,
			confirm: confirmator,
			notify:  notify,
			sock:    sock,
			logger:  logger,
		})
		if err != nil {
			return nil, fmt.Errorf("can't create socket %q: %w", sock.Name, err)
		}

		agents = append(agents, agent)
		realAgents[sock.Name] = i
		i++
	}

	// then - proxy agents
	for _, sock := range sockets {
		if sock.SameAs == "" {
			continue
		}

		parentID, ok := realAgents[sock.SameAs]
		if !ok {
			return nil, fmt.Errorf("can't create socket %q: unknown parent: %s", sock.Name, sock.SameAs)
		}

		agent, err := newProxyAgent(ProxyAgentParams{
			parent: agents[parentID],
			sock:   sock,
			logger: logger,
		})
		if err != nil {
			return nil, fmt.Errorf("can't create socket %q: %w", sock.Name, err)
		}

		agents = append(agents, agent)
	}

	return &AgentsGang{
		agents: agents,
	}, nil
}

func (a *AgentsGang) ListenAndServe() error {
	g, ctx := errgroup.WithContext(context.Background())
	for _, agent := range a.agents {
		agent := agent
		g.Go(func() error {
			if err := agent.ListenAndServe(ctx); err != nil {
				return fmt.Errorf("listen on socket %q failed: %w", agent.Name(), err)
			}

			return nil
		})
	}

	return g.Wait()
}

func (a *AgentsGang) ReloadKeys() error {
	for _, agent := range a.agents {
		agent.ReloadKeys()
	}

	return nil
}

func (a *AgentsGang) Shutdown(ctx context.Context) error {
	var wg sync.WaitGroup
	for _, agent := range a.agents {
		wg.Add(1)
		agent := agent
		go func() {
			defer wg.Done()

			agent.Shutdown(ctx)
		}()
	}

	wg.Wait()
	return nil
}
