package agentkey

import (
	"time"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"

	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil"
)

var _ Key = (*AddedKey)(nil)

type AddedKey struct {
	name        string
	legacyFp    string
	fingerprint string
	persistent  bool
	confirm     bool
	validBefore time.Time
	signer      ssh.Signer
	agentKey    *agent.Key
}

func AddedRawKey(key agent.AddedKey, persistent bool) (*AddedKey, error) {
	signer, err := ssh.NewSignerFromKey(key.PrivateKey)
	if err != nil {
		return nil, err
	}

	var validBefore time.Time
	if cert := key.Certificate; cert != nil {
		validBefore = time.Unix(int64(key.Certificate.ValidBefore), 0)
		signer, err = ssh.NewCertSigner(cert, signer)
		if err != nil {
			return nil, err
		}
	}

	pubKey := signer.PublicKey()
	fingerprint := ssh.FingerprintSHA256(pubKey)

	var name string
	if key.Comment != "" {
		name = key.Comment
	} else {
		name = sshutil.Fingerprint(pubKey)
	}

	return &AddedKey{
		name:        name,
		legacyFp:    ssh.FingerprintLegacyMD5(pubKey),
		fingerprint: fingerprint,
		persistent:  persistent,
		confirm:     key.ConfirmBeforeUse,
		validBefore: validBefore,
		signer:      signer,
		agentKey: &agent.Key{
			Format:  pubKey.Type(),
			Blob:    pubKey.Marshal(),
			Comment: key.Comment,
		},
	}, nil
}

func (k *AddedKey) Name() string {
	return k.name
}

func (k *AddedKey) Kind() KeyKind {
	return KeyKindAdded
}

func (k *AddedKey) LegacyFingerprint() string {
	return k.legacyFp
}

func (k *AddedKey) Fingerprint() string {
	return k.fingerprint
}

func (k *AddedKey) TouchNeeded() bool {
	return false
}

func (k *AddedKey) ConfirmBeforeUse() bool {
	return k.confirm
}

func (k *AddedKey) AgentKey() *agent.Key {
	return k.agentKey
}

func (k *AddedKey) Persistent() bool {
	return k.persistent
}

func (k *AddedKey) IsExpired() bool {
	return !k.validBefore.IsZero() && time.Now().After(k.validBefore)
}

func (k *AddedKey) ValidBefore() time.Time {
	return k.validBefore
}

func (k *AddedKey) IsOurKey() bool {
	return false
}

func (k *AddedKey) Sign(data []byte, flags agent.SignatureFlags) (*ssh.Signature, error) {
	return sign(k.signer, data, flags)
}
