package agentkey

import (
	"crypto/rand"
	"fmt"
	"time"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"
)

type Key interface {
	Name() string
	Kind() KeyKind
	LegacyFingerprint() string
	Fingerprint() string
	TouchNeeded() bool
	ConfirmBeforeUse() bool
	AgentKey() *agent.Key
	Persistent() bool
	IsOurKey() bool
	IsExpired() bool
	ValidBefore() time.Time
	Sign(data []byte, flags agent.SignatureFlags) (*ssh.Signature, error)
}

func sign(signer ssh.Signer, data []byte, flags agent.SignatureFlags) (*ssh.Signature, error) {
	if flags == 0 {
		return signer.Sign(rand.Reader, data)
	}

	algorithmSigner, ok := signer.(ssh.AlgorithmSigner)
	if !ok {
		return nil, fmt.Errorf("signature does not support non-default signature algorithm: %T", signer)
	}

	var algorithm string
	switch flags {
	case agent.SignatureFlagRsaSha256:
		algorithm = ssh.KeyAlgoRSASHA256
	case agent.SignatureFlagRsaSha512:
		algorithm = ssh.KeyAlgoRSASHA512
	default:
		return nil, fmt.Errorf("unsupported signature flags: %d", flags)
	}

	return algorithmSigner.SignWithAlgorithm(rand.Reader, data, algorithm)
}
