package agentkey

import (
	"fmt"
	"sync"
	"time"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"

	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/pubstore"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil"
)

var _ Key = (*KeyringKey)(nil)

type KeyringKey struct {
	ks          keyring.Keyring
	kind        KeyKind
	keyPurpose  keyring.KeyPurpose
	name        string
	legacyFp    string
	fingerprint string
	touchNeeded bool
	confirm     bool
	validBefore time.Time
	agentKey    *agent.Key
	mu          sync.Mutex
}

func NewKeyringKey(kr keyring.Keyring, ps pubstore.PubStore, keyPurpose keyring.KeyPurpose, confirm bool) (*KeyringKey, error) {
	var kind KeyKind
	if err := kind.FromKeyPurpose(keyPurpose); err != nil {
		return nil, fmt.Errorf("unable to determive key kind: %w", err)
	}

	sshPubBytes, err := ps.ReadKey(kr.Name(), keyPurpose)
	if err != nil {
		return nil, fmt.Errorf("failed to read ssh pub key: %w", err)
	}

	sshPub, _, _, _, err := ssh.ParseAuthorizedKey(sshPubBytes)
	if err != nil {
		return nil, fmt.Errorf("failed to parse ssh pub key %q: %w", keyPurpose, err)
	}

	var validBefore time.Time
	if cert, ok := sshPub.(*ssh.Certificate); ok {
		validBefore = time.Unix(int64(cert.ValidBefore), 0)
	}

	out := &KeyringKey{
		ks:          kr,
		kind:        kind,
		keyPurpose:  keyPurpose,
		name:        fmt.Sprintf("%s@%s", keyPurpose, kr.Name()),
		legacyFp:    ssh.FingerprintLegacyMD5(sshPub),
		fingerprint: sshutil.Fingerprint(sshPub),
		touchNeeded: kr.IsTouchableKey(keyPurpose),
		validBefore: validBefore,
		confirm:     confirm,
		agentKey: &agent.Key{
			Format:  sshPub.Type(),
			Blob:    sshPub.Marshal(),
			Comment: fmt.Sprintf("Skotty key %s on %s", keyPurpose, kr.HumanName()),
		},
	}

	return out, nil
}

func (k *KeyringKey) Name() string {
	return k.name
}

func (k *KeyringKey) Kind() KeyKind {
	return k.kind
}

func (k *KeyringKey) LegacyFingerprint() string {
	return k.legacyFp
}

func (k *KeyringKey) Fingerprint() string {
	return k.fingerprint
}

func (k *KeyringKey) TouchNeeded() bool {
	return k.touchNeeded
}

func (k *KeyringKey) ConfirmBeforeUse() bool {
	return k.confirm
}

func (k *KeyringKey) AgentKey() *agent.Key {
	return k.agentKey
}

func (k *KeyringKey) Persistent() bool {
	return true
}

func (k *KeyringKey) IsOurKey() bool {
	return true
}

func (k *KeyringKey) IsExpired() bool {
	return !k.validBefore.IsZero() && time.Now().After(k.validBefore)
}

func (k *KeyringKey) ValidBefore() time.Time {
	return k.validBefore
}

func (k *KeyringKey) Sign(data []byte, flags agent.SignatureFlags) (*ssh.Signature, error) {
	tx, err := k.ks.Tx()
	if err != nil {
		return nil, fmt.Errorf("open keyring tx: %w", err)
	}
	defer tx.Close()

	keySigner, err := tx.Signer(k.keyPurpose)
	if err != nil {
		return nil, fmt.Errorf("get private key: %w", err)
	}

	sshSigner, err := ssh.NewSignerFromKey(keySigner)
	if err != nil {
		return nil, fmt.Errorf("create ssh signer: %w", err)
	}

	return sign(sshSigner, data, flags)
}
