package agentkey

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
)

var _ yaml.Unmarshaler = (*KeyKind)(nil)
var _ yaml.Marshaler = (*KeyKind)(nil)
var _ json.Unmarshaler = (*KeyKind)(nil)
var _ json.Marshaler = (*KeyKind)(nil)

type KeyKind uint8

const (
	KeyKindNone = iota
	KeyKindAdded
	KeyKindSudo
	KeyKindSecure
	KeyKindInsecure
	KeyKindLegacy
	KeyKindRenew
)

func (k KeyKind) String() string {
	switch k {
	case KeyKindNone:
		return "none"
	case KeyKindAdded:
		return "added"
	case KeyKindSecure:
		return "secure"
	case KeyKindInsecure:
		return "insecure"
	case KeyKindSudo:
		return "sudo"
	case KeyKindLegacy:
		return "legacy"
	case KeyKindRenew:
		return "renew"
	default:
		return fmt.Sprintf("key_%d", k)
	}
}

func (k *KeyKind) FromString(v string) error {
	switch v {
	case "none":
		*k = KeyKindNone
	case "added":
		*k = KeyKindAdded
	case "secure":
		*k = KeyKindSecure
	case "insecure":
		*k = KeyKindInsecure
	case "sudo":
		*k = KeyKindSudo
	case "renew":
		*k = KeyKindRenew
	case "legacy":
		*k = KeyKindLegacy
	default:
		return fmt.Errorf("unknown key kind: %s", v)
	}

	return nil
}

func (k *KeyKind) FromKeyPurpose(p keyring.KeyPurpose) error {
	switch p {
	case keyring.KeyPurposeNone:
		*k = KeyKindNone
	case keyring.KeyPurposeSecure:
		*k = KeyKindSecure
	case keyring.KeyPurposeInsecure:
		*k = KeyKindInsecure
	case keyring.KeyPurposeSudo:
		*k = KeyKindSudo
	case keyring.KeyPurposeLegacy:
		*k = KeyKindLegacy
	case keyring.KeyPurposeRenew:
		*k = KeyKindRenew
	default:
		return fmt.Errorf("unknown key purpose: %s", p)
	}

	return nil
}

func (k KeyKind) MarshalYAML() (interface{}, error) {
	return k.String(), nil
}

func (k *KeyKind) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return k.FromString(s)
}

func (k KeyKind) MarshalJSON() ([]byte, error) {
	return json.Marshal(k.String())
}

func (k *KeyKind) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return k.FromString(s)
}
