package agent

import (
	"context"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
)

var _ agent.ExtendedAgent = (*loggableHandler)(nil)

type loggableHandler struct {
	ctx     context.Context
	handler SSHAgent
	log     log.Logger
}

// List returns the identities known to the agent.
func (h *loggableHandler) List() ([]*agent.Key, error) {
	ctxlog.Debug(h.ctx, h.log, ".List called")
	out, err := h.handler.List(h.ctx)
	if err != nil {
		h.log.Error(".List fail", log.Error(err))
	}

	return out, err
}

// Sign has the agent sign the data using a protocol 2 key as defined
// in [PROTOCOL.agent] section 2.6.2.
func (h *loggableHandler) Sign(reqKey ssh.PublicKey, data []byte) (*ssh.Signature, error) {
	ctxlog.Debug(h.ctx, h.log, ".Sign called")
	out, err := h.handler.Sign(h.ctx, reqKey, data)
	if err != nil {
		h.log.Error(".Sign fail", log.Error(err))
	}

	return out, err
}

// SignWithFlags signs like Sign, but allows for additional flags to be sent/received
func (h *loggableHandler) SignWithFlags(reqKey ssh.PublicKey, data []byte, flags agent.SignatureFlags) (*ssh.Signature, error) {
	ctxlog.Debug(h.ctx, h.log, ".SignWithFlags called")
	out, err := h.handler.SignWithFlags(h.ctx, reqKey, data, flags)
	if err != nil {
		h.log.Error(".SignWithFlags fail", log.Error(err))
	}

	return out, err
}

// Extension processes a custom extension request. Standard-compliant agents are not
// required to support any extensions, but this method allows agents to implement
// vendor-specific methods or add experimental features. See [PROTOCOL.agent] section 4.7.
func (h *loggableHandler) Extension(extensionType string, contents []byte) ([]byte, error) {
	ctxlog.Debug(h.ctx, h.log, ".Extension called")
	out, err := h.handler.Extension(h.ctx, extensionType, contents)
	if err != nil && err != agent.ErrExtensionUnsupported {
		h.log.Error(".Extension fail",
			log.String("extension", extensionType),
			log.Error(err),
		)
	}

	return out, err
}

// Signers returns signers for all the known keys.
func (h *loggableHandler) Signers() ([]ssh.Signer, error) {
	ctxlog.Debug(h.ctx, h.log, ".Signers called")
	out, err := h.handler.Signers(h.ctx)
	if err != nil {
		h.log.Error(".Signers fail", log.Error(err))
	}

	return out, err
}

// Add adds a private key to the agent.
func (h *loggableHandler) Add(newKey agent.AddedKey) error {
	ctxlog.Debug(h.ctx, h.log, ".Add called")
	err := h.handler.Add(h.ctx, newKey)
	if err != nil {
		h.log.Error(".Add fail", log.String("key_name", newKey.Comment), log.Error(err))
	}

	return err
}

// Remove removes all identities with the given public key.
func (h *loggableHandler) Remove(reqKey ssh.PublicKey) error {
	ctxlog.Debug(h.ctx, h.log, ".Remove called")
	err := h.handler.Remove(h.ctx, reqKey)
	if err != nil {
		h.log.Error(".Remove fail", log.Error(err))
	}

	return err
}

// RemoveAll removes all identities.
func (h *loggableHandler) RemoveAll() error {
	ctxlog.Debug(h.ctx, h.log, ".RemoveAll called")
	err := h.handler.RemoveAll(h.ctx)
	if err != nil {
		h.log.Error(".RemoveAll fail", log.Error(err))
	}

	return err
}

// Lock locks the agent. Sign and Remove will fail, and List will empty an empty list.
func (h *loggableHandler) Lock(passphrase []byte) error {
	ctxlog.Debug(h.ctx, h.log, ".Lock called")
	err := h.handler.Lock(h.ctx, passphrase)
	if err != nil {
		h.log.Error(".Lock fail", log.Error(err))
	}

	return err
}

// Unlock undoes the effect of Lock
func (h *loggableHandler) Unlock(passphrase []byte) error {
	ctxlog.Debug(h.ctx, h.log, ".Unlock called")
	err := h.handler.Unlock(h.ctx, passphrase)
	if err != nil {
		h.log.Error(".Unlock fail", log.Error(err))
	}

	return err
}
