package agent

import (
	"context"
	"os"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/skotty/internal/config"
)

var _ Agent = (*ProxyAgent)(nil)

type ProxyAgent struct {
	listener
	name   string
	parent Agent
	log    log.Logger
}

type ProxyAgentParams struct {
	parent Agent
	sock   config.Socket
	logger log.Logger
}

func newProxyAgent(params ProxyAgentParams) (*ProxyAgent, error) {
	l := params.logger.WithName(params.sock.Name)
	out := &ProxyAgent{
		name:   params.sock.Name,
		parent: params.parent,
		log:    l,
	}

	out.listener = listener{
		uid:     os.Getuid(),
		kind:    params.sock.Kind,
		addr:    params.sock.Path,
		handler: out,
		log:     l,
	}

	return out, nil
}

func (a *ProxyAgent) Name() string {
	return a.name
}

func (a *ProxyAgent) ReloadKeys() {}

func (a *ProxyAgent) List(ctx context.Context) ([]*agent.Key, error) {
	a.log.Info("proxying List request", log.String("dst", a.parent.Name()))
	return a.parent.List(ctx)
}

func (a *ProxyAgent) Sign(ctx context.Context, key ssh.PublicKey, data []byte) (*ssh.Signature, error) {
	a.log.Info("proxying Sign request", log.String("dst", a.parent.Name()))
	return a.parent.Sign(ctx, key, data)
}

func (a *ProxyAgent) Add(ctx context.Context, key agent.AddedKey) error {
	a.log.Info("proxying Add request", log.String("dst", a.parent.Name()))
	return a.parent.Add(ctx, key)
}

func (a *ProxyAgent) Remove(ctx context.Context, key ssh.PublicKey) error {
	a.log.Info("proxying Remove request", log.String("dst", a.parent.Name()))
	return a.parent.Remove(ctx, key)
}

// RemoveAll removes all identities.
func (a *ProxyAgent) RemoveAll(ctx context.Context) error {
	a.log.Info("proxying RemoveAll request", log.String("dst", a.parent.Name()))
	return a.parent.RemoveAll(ctx)
}

// Lock locks the agent. Sign and Remove will fail, and List will empty an empty list.
func (a *ProxyAgent) Lock(ctx context.Context, passphrase []byte) error {
	a.log.Info("proxying Lock request", log.String("dst", a.parent.Name()))
	return a.parent.Lock(ctx, passphrase)
}

func (a *ProxyAgent) Unlock(ctx context.Context, passphrase []byte) error {
	a.log.Info("proxying Unlock request", log.String("dst", a.parent.Name()))
	return a.parent.Unlock(ctx, passphrase)
}

func (a *ProxyAgent) Signers(ctx context.Context) ([]ssh.Signer, error) {
	a.log.Info("proxying Signers request", log.String("dst", a.parent.Name()))
	return a.parent.Signers(ctx)
}

func (a *ProxyAgent) SignWithFlags(ctx context.Context, key ssh.PublicKey, data []byte, flags agent.SignatureFlags) (*ssh.Signature, error) {
	a.log.Info("proxying SignWithFlags request", log.String("dst", a.parent.Name()))
	return a.parent.SignWithFlags(ctx, key, data, flags)
}

func (a *ProxyAgent) Extension(ctx context.Context, extensionType string, contents []byte) ([]byte, error) {
	a.log.Info("proxying Extension request", log.String("dst", a.parent.Name()))
	return a.parent.Extension(ctx, extensionType, contents)
}
