package certgen

import (
	"crypto"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"crypto/rsa"
	"crypto/x509"
	"crypto/x509/pkix"
	"fmt"
	"math/big"
	"time"

	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
)

func GenCertificate(keyPurpose keyring.KeyPurpose) ([]byte, []byte, error) {
	var privKey crypto.Signer
	var privKeyBytes []byte
	switch keyPurpose {
	case keyring.KeyPurposeSudo, keyring.KeyPurposeInsecure, keyring.KeyPurposeSecure:
		key, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		if err != nil {
			return nil, nil, fmt.Errorf("failed to generate private key: %w", err)
		}

		privKey = key
		privKeyBytes, err = x509.MarshalECPrivateKey(key)
		if err != nil {
			return nil, nil, fmt.Errorf("failed to marshal private key: %w", err)
		}
	case keyring.KeyPurposeLegacy:
		key, err := rsa.GenerateKey(rand.Reader, 2048)
		if err != nil {
			return nil, nil, fmt.Errorf("failed to generate private key: %w", err)
		}

		privKey = key
		privKeyBytes = x509.MarshalPKCS1PrivateKey(key)
	default:
		return nil, nil, fmt.Errorf("unsupported key purpose: %s", keyPurpose)
	}

	pubKeyBytes, err := GenCertificateFor(GenCommonName(keyPurpose), privKey.Public())
	if err != nil {
		return nil, nil, fmt.Errorf("unable to generate certificate: %w", err)
	}

	return pubKeyBytes, privKeyBytes, nil
}

func GenCertificateFor(commonName string, pub crypto.PublicKey) ([]byte, error) {
	var parentPriv crypto.PrivateKey
	var parentPub crypto.PublicKey
	switch pub.(type) {
	case *rsa.PublicKey:
		priv, err := rsa.GenerateKey(rand.Reader, 1024)
		if err != nil {
			return nil, fmt.Errorf("couldn't generate parent priv key: %w", err)
		}

		parentPub = priv.Public()
		parentPriv = priv
	case *ecdsa.PublicKey:
		priv, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		if err != nil {
			return nil, fmt.Errorf("couldn't generate parent priv key: %w", err)
		}

		parentPub = priv.Public()
		parentPriv = priv
	default:
		return nil, fmt.Errorf("unsupported pub key type: %T", pub)
	}

	parent := &x509.Certificate{
		Subject: pkix.Name{
			CommonName: "Skotty SSH CA (Fake)",
		},
		PublicKey: parentPub,
	}

	csr := &x509.Certificate{
		SerialNumber: big.NewInt(time.Now().UnixNano()),
		Subject: pkix.Name{
			CommonName: commonName,
		},
		NotBefore:             time.Now(),
		NotAfter:              time.Now().AddDate(25, 0, 0),
		ExtKeyUsage:           []x509.ExtKeyUsage{x509.ExtKeyUsageClientAuth},
		KeyUsage:              x509.KeyUsageDigitalSignature | x509.KeyUsageCertSign,
		BasicConstraintsValid: true,
	}

	certBytes, err := x509.CreateCertificate(rand.Reader, csr, parent, pub, parentPriv)
	if err != nil {
		return nil, fmt.Errorf("couldn't to generate certificate: %w", err)
	}

	return certBytes, nil
}

func GenCommonName(keyPurpose keyring.KeyPurpose) string {
	return fmt.Sprintf("skotty@%s", keyPurpose)
}
