package commands

import (
	"context"
	"fmt"
	"strings"
	"text/template"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/filering"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/keychain"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/yubiring"
	"a.yandex-team.ru/security/skotty/skotty/internal/version"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
)

const infoTmpl = `
--- Skotty ---
Version: {{.Version}}

--- Agent ---
Log: {{.Agent.LogPath}}
Status: {{.Agent.Status}}
{{if .Agent.Version}}Version: {{.Agent.Version}}{{if ne .Agent.Version .Version}} differs from the current version {{.Version}}. Probably you need to restart them{{end}}{{end}}
{{if .Agent.PID}}Pid: {{.Agent.PID}}{{end}}

--- Sockets ---
{{- range $sock := .Sockets}}
- {{$sock.Name}}
  * path: {{$sock.Path}}
  * kind: {{$sock.Kind}}
{{if $sock.SameAs}}  * same as: {{$sock.SameAs}}{{else}}  * keys: {{$sock.Keys}}{{end}}
{{- end}}

--- Keyring ---
ID: {{.KeyringID}}
Kind: {{.Keyring}}
{{.KeyringInfo}}
--- Keys ---
{{- range $key := .Keys}}
{{- if $key.CA}}
- {{$key.Name}}:{{if $key.Err}} unable to get key info: {{$key.Err}}{{end}}
  * expires: {{$key.Expires}}
  * CA: {{$key.CA}}
{{- else}}
- {{$key.Name}}{{if $key.Err}}: unable to get key info: {{$key.Err}}{{end}}
{{- end}}{{end}}
`

const (
	yubiringTmpl = `
Serial: {{.Serial}}
PIN provider: {{.PIN}}
`
	keychainTmpl = `
Collection: {{.Collection}}
`
	fileringTmpl = `
Base path: {{.BasePath}}
Passphrase provider: {{.Passphrase}}
`
)

var infoCmd = &cobra.Command{
	Use:          "info",
	SilenceUsage: true,
	Short:        "Prints detailed information about yubikeys",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		type Socket struct {
			Name   string
			Path   string
			Kind   string
			SameAs string
			Keys   string
		}

		type Key struct {
			Name    string
			Expires string
			CA      string
			Err     string
		}

		type Agent struct {
			LogPath string
			Status  string
			Version string
			PID     int
		}

		infoData := struct {
			Version     string
			Agent       Agent
			Sockets     []Socket
			KeyringID   string
			Keyring     string
			KeyringInfo string
			Keys        []Key
		}{
			Version: version.Full(),
		}

		render := func(tmpl string, data interface{}) string {
			var out strings.Builder
			err := template.Must(template.New("tmpl").Parse(tmpl)).Execute(&out, data)
			if err != nil {
				panic(fmt.Sprintf("can't render template: %v", err))
			}

			return out.String()
		}

		for _, sock := range cfg.Sockets {
			var keysInfo strings.Builder
			for k, key := range sock.Keys {
				if k != 0 {
					keysInfo.WriteString(" + ")
				}
				keysInfo.WriteString(key.String())
			}

			infoData.Sockets = append(infoData.Sockets, Socket{
				Name:   sock.Name,
				Path:   sock.Path,
				SameAs: sock.SameAs,
				Kind:   sock.Kind.String(),
				Keys:   keysInfo.String(),
			})
		}

		infoData.Agent = func() Agent {
			out := Agent{
				LogPath: cfg.AgentLogPath,
				Status:  "n/a",
			}

			sc, err := skottyctl.NewClient(cfg.CtlSocketPath)
			if err != nil {
				return out
			}
			defer func() { _ = sc.Close() }()

			ctx, cancel := context.WithTimeout(context.Background(), 500*time.Millisecond)
			defer cancel()

			status, err := sc.Status(ctx)
			if err != nil {
				return out
			}

			out.Version = status.Version
			out.PID = status.PID
			switch status.Status {
			case skottyctl.StatusOk:
				out.Status = "OK"
			case skottyctl.StatusKeyReloading:
				out.Status = "Reload keys"
			case skottyctl.StatusRestarting:
				out.Status = "Restarting"
			default:
				out.Status = status.Status.String()
			}

			return out
		}()

		kr, err := cfg.NewKeyring()
		if err != nil {
			return fmt.Errorf("can't open keyring: %w", err)
		}
		defer kr.Close()

		tx, err := kr.Tx()
		if err != nil {
			return fmt.Errorf("can't start keyring transaction: %w", err)
		}
		defer tx.Close()

		keyInfo := func(purpose keyring.KeyPurpose) Key {
			cert, err := tx.Certificate(purpose)
			if err != nil {
				return Key{
					Name: purpose.String(),
					Err:  err.Error(),
				}
			}

			key := Key{
				Name: purpose.String(),
			}

			if purpose != keyring.KeyPurposeLegacy {
				key.Expires = cert.NotAfter.Local().Format(time.RFC822)
				key.CA = cert.Issuer.CommonName
			}

			return key
		}

		for _, purpose := range cfg.Keyring.Keys {
			infoData.Keys = append(infoData.Keys, keyInfo(purpose))
		}

		infoData.KeyringID = func() string {
			out, err := kr.Serial()
			if err != nil {
				return err.Error()
			}

			id, err := skotty.SerialToTokenID(kr.TokenType(), out)
			if err != nil {
				return err.Error()
			}

			return id
		}()

		switch cfg.Keyring.Kind {
		case keyring.KindYubikey:
			yi := struct {
				Serial uint32
				PIN    string
			}{
				Serial: cfg.Keyring.Yubikey.Serial,
				PIN:    cfg.Keyring.Yubikey.PIN.Kind().String(),
			}

			infoData.Keyring = yubiring.HumanName
			infoData.KeyringInfo = render(yubiringTmpl[1:], yi)
		case keyring.KindKeychain:
			infoData.Keyring = keychain.HumanName
			infoData.KeyringInfo = render(keychainTmpl[1:], cfg.Keyring.Keychain)
		case keyring.KindFiles:
			ki := struct {
				BasePath   string
				Passphrase string
			}{
				BasePath:   cfg.Keyring.Files.BasePath,
				Passphrase: cfg.Keyring.Files.Passphrase.Kind().String(),
			}

			infoData.Keyring = filering.HumanName
			infoData.KeyringInfo = render(fileringTmpl[1:], ki)
		default:
			return fmt.Errorf("upsupported keyring: %s", cfg.Keyring.Kind)
		}

		out := render(infoTmpl[1:], infoData)
		fmt.Print(out)
		return nil
	},
}
