package commands

import (
	"context"
	"fmt"
	"os"
	"path/filepath"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/skotty/skotty/internal/assets"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
)

var notifyCmd = &cobra.Command{
	Use:          "notify",
	SilenceUsage: true,
	Short:        "Notify skotty about something",
}

var notifyStartupTTYCmd = &cobra.Command{
	Use:          "startup-tty",
	SilenceUsage: true,
	Short:        "Set startup TTY and X11 DISPLAY variables to the values of this session.",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		ctl, err := skottyctl.NewClient(cfg.CtlSocketPath)
		if err != nil {
			return fmt.Errorf("unable to create skottyctl client: %w", err)
		}
		defer func() { _ = ctl.Close() }()

		if err := ctl.UpdateStartupTTY(context.Background()); err != nil {
			return err
		}

		return nil
	},
}

var notifyInstallCmd = &cobra.Command{
	Use:          "install",
	Hidden:       true,
	SilenceUsage: true,
	Short:        "Notify skotty to complete installation process",
	RunE: func(_ *cobra.Command, _ []string) error {
		selfExe, err := os.Executable()
		if err != nil {
			return fmt.Errorf("failed to determine self executable path: %w", err)
		}

		if err := assets.UnBundle(selfExe, filepath.Dir(selfExe)); err != nil {
			return fmt.Errorf("can't unbundle skotty resources: %w", err)
		}

		return nil
	},
}

var notifyReloadKeysCmd = &cobra.Command{
	Use:          "reload-keys",
	SilenceUsage: true,
	Hidden:       true,
	Short:        "Signal skotty to reload keys.",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		ctl, err := skottyctl.NewClient(cfg.CtlSocketPath)
		if err != nil {
			return fmt.Errorf("unable to create skottyctl client: %w", err)
		}
		defer func() { _ = ctl.Close() }()

		if err := ctl.ReloadKeys(context.Background()); err != nil {
			return err
		}

		fmt.Println("ok")
		return nil
	},
}

var notifyRestartCmd = &cobra.Command{
	Use:          "restart",
	SilenceUsage: true,
	Hidden:       true,
	Short:        "Signal skotty to restart.",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		ctl, err := skottyctl.NewClient(cfg.CtlSocketPath)
		if err != nil {
			return fmt.Errorf("unable to create skottyctl client: %w", err)
		}
		defer func() { _ = ctl.Close() }()

		if err := ctl.Restart(context.Background()); err != nil {
			return err
		}

		fmt.Println("ok")
		return nil
	},
}

func init() {
	notifyCmd.AddCommand(
		notifyStartupTTYCmd,
		notifyInstallCmd,
		notifyReloadKeysCmd,
		notifyRestartCmd,
	)
}
