package commands

import (
	"context"
	"crypto/x509"
	"fmt"
	"os"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/pubstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/scenario"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
)

var renewCmd = &cobra.Command{
	Use:          "renew",
	SilenceUsage: true,
	Short:        "Renewing certificates",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfgPath, err := paths.Config()
		if err != nil {
			return err
		}

		cfg, err := config.Load(cfgPath, true)
		if err != nil {
			return fmt.Errorf("[-] Can't read config: %w", err)
		}

		s := scenario.NewRenew(
			scenario.WithEnrollUpstream(cfg.EnrollmentService),
			scenario.WithPubStore(pubstore.NewFileStore(cfg.SSHKeysPath)),
		)

		doRenew := func() error {
			s.LogSuccess("used keyring: %s", cfg.Keyring.Kind.HumanName())
			kr, err := cfg.NewKeyring()
			if err != nil {
				return fmt.Errorf("can't open keyring: %w", err)
			}
			defer kr.Close()

			renewInfo := scenario.RenewInfo{
				RenewToken: cfg.Keyring.RenewToken.String(),
				EnrollInfo: cfg.Keyring.EnrollInfo,
			}

			return s.CollectCerts(kr, func(oldCerts []*x509.Certificate) error {
				return s.Request(kr, renewInfo, func(renewRsp *skotty.RequestRenewRsp) error {
					webauth := func(_ string, next func() error) error {
						return next()
					}

					err := s.Approve(kr, renewRsp)
					if err != nil {
						s.LogError("fail: %v", err)
						s.LogSuccess("to complete the renew request, you must to perform two-factor authentication via URL: %s ", renewRsp.AuthURL)
						webauth = s.OpenInBrowser
					}

					return webauth(renewRsp.AuthURL, func() error {
						return s.RenewCerts(kr, func(certs []scenario.TokenCert) error {
							issueReq := scenario.IssueRenewReq{
								EnrollID:  renewRsp.EnrollmentID,
								AuthToken: renewRsp.AuthToken,
								Certs:     certs,
							}

							return s.WaitAndIssue(kr, issueReq, func(renew *scenario.IssuedRenew) error {
								s.LogSuccess("got %d certs", len(renew.Certs))
								s.LogWarn("certificates expires at: %s", renew.ExpiresAt.Format(time.RFC822))

								return s.UpdateKeys(kr, cfg.Keyring.Keys, renew.Certs, func(keys ...keyring.KeyPurpose) error {
									strKeys := make([]string, len(keys))
									for i, k := range keys {
										strKeys[i] = " " + k.String()
									}

									s.LogSuccess("certificates updated")
									cfg.Keyring.EnrollInfo = renew.EnrollInfo
									cfg.Keyring.Keys = keys
									//TODO(buglloc): drop me after 3 month
									cfg.Keyring.RenewToken = ""

									return s.CleanupCerts(oldCerts, func() error {
										return nil
									})
								})
							})
						})
					})
				})
			})
		}

		saveConfig := func() error {
			s.LogInfo("save config")

			err = config.Save(cfg, cfgPath)
			if err != nil {
				return err
			}

			s.LogSuccess("config saved into: %s", cfgPath)
			return nil
		}

		reloadKeys := func() error {
			s.LogInfo("reload keys")

			ctl, err := skottyctl.NewClient(cfg.CtlSocketPath)
			if err != nil {
				return err
			}

			err = ctl.ReloadKeys(context.Background())
			if err != nil {
				s.LogInfo("skotty not started")
				return nil
			}

			s.LogSuccess("keys reloaded")
			return nil
		}

		steps := []func() error{
			doRenew,
			saveConfig,
			reloadKeys,
		}
		for _, step := range steps {
			if err := step(); err != nil {
				s.LogError("fail: %v", err)
				os.Exit(1)
			}
		}

		fmt.Println("Done")
		return nil
	},
}
