package commands

import (
	"context"
	"errors"
	"fmt"
	"os"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/sectools"
	"a.yandex-team.ru/security/skotty/libs/isroot"
	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
	"a.yandex-team.ru/security/skotty/skotty/internal/version"
	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
)

var rootArgs struct {
	CheckLauncher bool
	AllowRoot     bool
}

var rootCmd = &cobra.Command{
	Use:           "skotty",
	SilenceUsage:  true,
	SilenceErrors: true,
	Short:         "Skotty - SSH Agent for Yandex",
	Long: `Skotty - yet another SSH Agent

Documentation: https://docs.yandex-team.ru/skotty/
Support chat: https://skotty.sec.yandex-team.ru/support/telegram/
In case of trouble run "skotty dump debug" and email to security@yandex-team.ru
`,
	PersistentPreRunE: func(cmd *cobra.Command, _ []string) error {
		if rootArgs.CheckLauncher {
			fmt.Println("nope")
			os.Exit(1)
		}

		if isroot.IsRoot() {
			if _, haveEnv := os.LookupEnv("SKOTTY_ALLOW_ROOT"); !haveEnv && !rootArgs.AllowRoot {
				return errors.New("you're trying to launch Skotty from the privileged user, most likely this is not correct. Otherwise, use --allow-root flag")
			}
		}

		if version.IsDirty() {
			// no need to check version of dirty build
			return nil
		}

		switch cmd.Name() {
		case sshEnvCmd.Name(), startCmd.Name(), notifyStartupTTYCmd.Name():
			return nil
		}

		ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
		defer cancel()

		secToolsOpts := []sectools.Option{
			sectools.WithPreferFastURL(),
			sectools.WithCurrentVersion(version.Full()),
		}

		if c, err := sectools.ParseChannel(os.Getenv("SKOTTY_CHANNEL")); err == nil {
			secToolsOpts = append(secToolsOpts, sectools.WithChannel(c))
		}

		svc := sectools.NewClient(version.ToolName, secToolsOpts...)
		isLatest, latestVersion, err := svc.IsLatestVersion(ctx, version.Full())
		if !isLatest && err == nil {
			var help string
			if osutil.UnderLauncher() {
				help = "please update me: skotty --update\n"
			}

			_, _ = fmt.Fprintf(os.Stderr, "-----\nNew Skotty version available (cur v%s): v%s\n%s-----\n", version.Full(), latestVersion, help)
		}

		return nil
	},
}

func Execute() error {
	return rootCmd.Execute()
}

func init() {
	cobra.EnableCommandSorting = false
	// hack to execute PersistentPreRunE
	rootCmd.RunE = func(_ *cobra.Command, _ []string) error {
		return rootCmd.Usage()
	}

	flags := rootCmd.PersistentFlags()
	flags.BoolVar(&rootArgs.CheckLauncher, "is-launcher", false, "check if this program is skotty launcher")
	flags.BoolVar(&rootArgs.AllowRoot, "allow-root", false, "allow starts from root")
	_ = flags.MarkHidden("allow-root")

	if osutil.UnderLauncher() {
		_ = flags.String("set-channel", "", "set distribution channel (stable/prestable/testing)")
		_ = flags.Bool("update", false, "update skotty")
		_ = flags.Bool("has-update", false, "checks skotty update")
	}

	rootCmd.AddCommand(
		startCmd,
		setupCmd,
		renewCmd,
		infoCmd,
		sshCmd,
		yubikeyCmd,
		versionCmd,
		serviceCmd,
		notifyCmd,
		passtoolCmd,
		dumpCmd,
	)
}

func loadConfig(strict bool) (*config.Config, error) {
	cfgPath, err := paths.Config()
	if err != nil {
		return nil, err
	}

	return config.Load(cfgPath, strict)
}
