package commands

import (
	"context"
	"errors"
	"fmt"
	"os"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
)

var serviceCmd = &cobra.Command{
	Use:          "service",
	SilenceUsage: true,
	Short:        "Configure skotty service",
}

var serviceEnableCmd = &cobra.Command{
	Use:          "enable",
	SilenceUsage: true,
	Short:        "Enable skotty service",
	RunE: func(_ *cobra.Command, _ []string) error {
		_, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		startCmd, err := osutil.InstallService()
		if err != nil {
			return err
		}

		if startCmd == "" {
			fmt.Println("skotty service successfully installed and probably started")
			return nil
		}
		fmt.Println("skotty service successfully installed, to start run:")
		fmt.Println(startCmd)
		return nil
	},
}

var serviceDisableCmd = &cobra.Command{
	Use:          "disable",
	SilenceUsage: true,
	Short:        "Disable skotty service",
	RunE: func(_ *cobra.Command, _ []string) error {
		err := osutil.UnInstallService()
		if err != nil {
			return err
		}
		fmt.Println("done")
		return nil
	},
}

var serviceRestartArgs struct {
	NoWait bool
}

var serviceRestartCmd = &cobra.Command{
	Use:          "restart",
	SilenceUsage: true,
	Short:        "Restarts skotty service",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		err = osutil.Restart()
		if err != nil {
			if errors.Is(err, osutil.ErrNotSupported) || errors.Is(err, osutil.ErrNotInstalled) {
				fmt.Println("nothing to do: skotty service is not installed")
				return nil
			}
			return err
		}

		if !serviceRestartArgs.NoWait {
			fmt.Println("wait Skotty to start...")

			err = backoff.Retry(
				func() error {
					sc, err := skottyctl.NewClient(cfg.CtlSocketPath)
					if err != nil {
						return fmt.Errorf("create client: %w", err)
					}
					defer func() { _ = sc.Close() }()

					status, err := sc.Status(context.Background())
					if err != nil {
						return fmt.Errorf("call skotty RPC: %w", err)
					}

					if status.Status != skottyctl.StatusOk {
						return fmt.Errorf("unexpected status: %s", status.Status.String())
					}

					return nil
				},
				backoff.WithMaxRetries(backoff.NewConstantBackOff(1*time.Second), 10),
			)

			if err != nil {
				return fmt.Errorf("skotty unable to start, client status: %w", err)
			}
		}

		fmt.Println("done")
		return nil
	},
}

var serviceStatusCmd = &cobra.Command{
	Use:          "status",
	SilenceUsage: true,
	Short:        "Check skotty status",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(false)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		checkSkottyStatus := func() string {
			sc, err := skottyctl.NewClient(cfg.CtlSocketPath)
			if err != nil {
				return fmt.Sprintf("create client: %s", err)
			}
			defer func() { _ = sc.Close() }()

			status, err := sc.Status(context.Background())
			if err != nil {
				return fmt.Sprintf("call skotty RPC: %s", err)
			}

			switch status.Status {
			case skottyctl.StatusOk:
				return fmt.Sprintf("OK (pid: %d)", status.PID)
			case skottyctl.StatusKeyReloading:
				return fmt.Sprintf("Reload keys (pid: %d)", status.PID)
			default:
				return fmt.Sprintf("n/a (pid: %d)", status.PID)
			}
		}

		fmt.Printf("skotty status: %s\n", checkSkottyStatus())
		if err := osutil.CheckService(); err != nil {
			fmt.Printf("skotty service: %s\n", err)
			os.Exit(1)
		}

		fmt.Println("skotty service: installed")
		return nil
	},
}

func init() {
	serviceRestartFlags := serviceRestartCmd.PersistentFlags()
	serviceRestartFlags.BoolVar(&serviceRestartArgs.NoWait, "no-wait", false, "no wait Skotty startup")

	serviceCmd.AddCommand(
		serviceEnableCmd,
		serviceDisableCmd,
		serviceRestartCmd,
		serviceStatusCmd,
	)
}
