package commands

import (
	"crypto/x509"
	"errors"
	"fmt"
	"os"
	"path/filepath"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/pubstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/anypin"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/scenario"
	"a.yandex-team.ru/security/skotty/skotty/internal/socket"
	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil/sshclient"
)

var setupArgs struct {
	Reuse           bool
	SocketsOnly     bool
	KeyringOnly     bool
	ConfigOnly      bool
	AllowAnyKeyring bool
}

var setupCmd = &cobra.Command{
	Use:          "setup",
	SilenceUsage: true,
	Short:        "Setup agent stuff (sockets, keyring, etc)",
	RunE: func(_ *cobra.Command, _ []string) error {
		if ok := checkTerm(); !ok {
			return nil
		}

		cfgPath, err := paths.Config()
		if err != nil {
			return fmt.Errorf("failed to determine config path: %w", err)
		}

		err = os.MkdirAll(filepath.Dir(cfgPath), 0o700)
		if err != nil {
			return fmt.Errorf("failed to create directory for configs: %w", err)
		}

		if setupArgs.Reuse && setupArgs.AllowAnyKeyring {
			return errors.New("can't use --reuse-yubikey and --any-keyring option at the same time")
		}

		var cfg *config.Config
		switch {
		case setupArgs.SocketsOnly, setupArgs.KeyringOnly, setupArgs.ConfigOnly:
			cfg, err = config.Load(cfgPath, true)
			if err != nil {
				return fmt.Errorf("failed to load config: %w", err)
			}
		default:
			cfg, err = config.Load("", false)
			if err != nil {
				return fmt.Errorf("failed to create new config: %w", err)
			}
		}

		s := scenario.NewSetup(
			scenario.WithEnrollUpstream(cfg.EnrollmentService),
			scenario.WithPubStore(pubstore.NewFileStore(cfg.SSHKeysPath)),
		)

		var staffKeys []scenario.AuthorizedKey
		staffKeysUpdated := false
		initializeToken := func() error {
			return s.SelectKeyring(!setupArgs.AllowAnyKeyring, func(kind keyring.Kind) error {
				s.LogSuccess("used keyring: %s", kind.HumanName())
				cfg.Keyring.Kind = kind

				initKeyring := func(next func(keyring.Keyring) error) error {
					return s.SelectPinstore(kind, func(pinProvider pinstore.Provider) error {
						switch kind {
						case keyring.KindYubikey:
							return s.NewYubikey(func(conf scenario.YubiringConf, kr keyring.Keyring) error {
								maskedPIN, maskedPUK := conf.MaskedPIN(), conf.MaskedPUK()
								if maskedPIN != "" {
									s.LogInfo("Yubikey PIN: %s", maskedPIN)
								}
								if maskedPUK != "" {
									s.LogInfo("Yubikey PUK: %s", maskedPUK)
								}

								if pinProvider.CanStore() {
									err = pinProvider.StorePIN(conf.PIN, kr.PinStoreOpts()...)
									if err != nil {
										return fmt.Errorf("unable to store pin: %w", err)
									}
								}

								cfg.Keyring.Yubikey = config.KeyringYubikey{
									Serial: conf.Serial,
									PIN: &anypin.Provider{
										Provider: pinProvider,
									},
								}
								return next(kr)
							})
						case keyring.KindKeychain:
							return s.NewKeychain(func(conf scenario.KeychainConf, kr keyring.Keyring) error {
								cfg.Keyring.Keychain = config.KeyringKeychain{
									Collection: conf.Collection,
								}
								return next(kr)
							})
						case keyring.KindFiles:
							return s.NewFilering(func(conf scenario.FileringConf, kr keyring.Keyring) error {
								if pinProvider.CanStore() {
									err = pinProvider.StorePassphrase(conf.Passphrase, kr.PinStoreOpts()...)
									if err != nil {
										return fmt.Errorf("unable to store passphrase: %w", err)
									}
								}

								cfg.Keyring.Files = config.KeyringFiles{
									BasePath: conf.BasePath,
									Passphrase: &anypin.Provider{
										Provider: pinProvider,
									},
								}

								return next(kr)
							})
						default:
							return fmt.Errorf("unsupported keyring: %s", kind)
						}
					})
				}

				return initKeyring(func(kr keyring.Keyring) error {
					return s.GenKeys(kr, func(temporaryCerts []scenario.TokenCert) error {
						return s.Request(kr, func(enrollRsp *skotty.RequestEnrollmentRsp) error {
							s.LogSuccess("to complete the enrollment request, you must to perform two-factor authentication via URL: %s ", enrollRsp.AuthURL)
							return s.OpenInBrowser(enrollRsp.AuthURL, func() error {
								issueReq := scenario.IssueEnrollmentReq{
									EnrollID:  enrollRsp.EnrollmentID,
									AuthToken: enrollRsp.AuthToken,
									Certs:     temporaryCerts,
								}

								return s.WaitAndIssue(kr, issueReq, func(enrollment *scenario.IssuedEnrollment) error {
									s.LogSuccess("got %d certs", len(enrollment.Certs))
									s.LogWarn("certificates expires at: %s", enrollment.ExpiresAt.Format(time.RFC822))

									switch {
									case enrollment.StaffUploaded:
										staffKeysUpdated = true
										s.LogSuccess("SSH keys updated on staff")
									case enrollment.StaffErr != "":
										// we must to check staff err for backward compatibility with old enrollment service
										s.LogError("can't update staff SSH keys (you must to add it manually later): %s", enrollment.StaffErr)
									}

									return s.UpdateKeys(kr, enrollment.Certs, func(keys ...keyring.KeyPurpose) error {
										strKeys := make([]string, len(keys))
										for i, k := range keys {
											strKeys[i] = " " + k.String()
										}

										s.LogSuccess("stored certificates:%s", strings.Join(strKeys, ","))
										cfg.Keyring.Keys = keys
										cfg.Keyring.EnrollInfo = enrollment.EnrollInfo
										//TODO(buglloc): drop me after 3 month
										cfg.Keyring.RenewToken = ""

										return s.AuthorizedKeys(kr.Name(), keys, func(authorizedKeys []scenario.AuthorizedKey) error {
											staffKeys = authorizedKeys

											oldCerts := make([]*x509.Certificate, len(temporaryCerts))
											for i, cert := range temporaryCerts {
												oldCerts[i] = cert.Cert
											}
											return s.CleanupCerts(oldCerts, func() error {
												return nil
											})
										})
									})
								})
							})
						})
					})
				})
			})
		}

		restoreToken := func() error {
			return s.SelectKeyring(true, func(kind keyring.Kind) error {
				s.LogSuccess("used keyring: %s", kind.HumanName())
				cfg.Keyring.Kind = kind

				initKeyring := func(next func(keyring.Keyring) error) error {
					return s.SelectPinstore(kind, func(pinProvider pinstore.Provider) error {
						switch kind {
						case keyring.KindYubikey:
							return s.OpenYubikey(func(conf scenario.YubiringConf, kr keyring.Keyring) error {
								if pinProvider.CanStore() {
									err = pinProvider.StorePIN(conf.PIN, kr.PinStoreOpts()...)
									if err != nil {
										return fmt.Errorf("unable to store pin: %w", err)
									}
								}

								cfg.Keyring.Yubikey = config.KeyringYubikey{
									Serial: conf.Serial,
									PIN: &anypin.Provider{
										Provider: pinProvider,
									},
								}
								return next(kr)
							})
						default:
							return fmt.Errorf("unsupported keyring: %s", kind)
						}
					})
				}

				return initKeyring(func(kr keyring.Keyring) error {
					sr := scenario.NewRenew(
						scenario.WithEnrollUpstream(cfg.EnrollmentService),
						scenario.WithPubStore(pubstore.NewFileStore(cfg.SSHKeysPath)),
					)

					var renewInfo scenario.RenewInfo
					return sr.RestoreKeys(kr, func(restoredKeys ...keyring.KeyPurpose) error {
						return sr.Request(kr, renewInfo, func(renewRsp *skotty.RequestRenewRsp) error {
							err := sr.Approve(kr, renewRsp)
							if err != nil {
								sr.LogError("fail: %v", err)
								sr.LogWarn("please authorize via URL: %s", renewRsp.AuthURL)
							}

							return sr.RenewCerts(kr, func(certs []scenario.TokenCert) error {
								issueReq := scenario.IssueRenewReq{
									EnrollID:  renewRsp.EnrollmentID,
									AuthToken: renewRsp.AuthToken,
									Certs:     certs,
								}

								return sr.WaitAndIssue(kr, issueReq, func(renew *scenario.IssuedRenew) error {
									sr.LogSuccess("got %d certs", len(renew.Certs))
									sr.LogWarn("certificates expires at: %s", renew.ExpiresAt.Format(time.RFC822))

									return sr.UpdateKeys(kr, restoredKeys, renew.Certs, func(keys ...keyring.KeyPurpose) error {
										strKeys := make([]string, len(keys))
										for i, k := range keys {
											strKeys[i] = " " + k.String()
										}

										sr.LogSuccess("certificates updated")
										staffKeysUpdated = true
										cfg.Keyring.EnrollInfo = renew.EnrollInfo
										cfg.Keyring.Keys = keys
										//TODO(buglloc): drop me after 3 month
										cfg.Keyring.RenewToken = ""
										return nil
									})
								})
							})
						})
					})
				})
			})
		}

		setupToken := func() error {
			if setupArgs.Reuse {
				return restoreToken()
			}

			return initializeToken()
		}

		initializeSockets := func() error {
			return s.SelectSockets(func(sockets []config.Socket) error {
				cfg.Sockets = sockets
				cfg.SSHAuthSock = sshclient.BestClient().SocketName(socket.NameDefault)
				return nil
			})
		}

		saveConfig := func() error {
			s.LogInfo("save config")

			err = config.Save(cfg, cfgPath)
			if err != nil {
				return err
			}

			s.LogSuccess("config saved into: %s", cfgPath)
			return nil
		}

		configureService := func() error {
			s.LogInfo("initialize service")

			err := s.InstallService(func(startCmd string) error {
				if startCmd == "" {
					s.LogSuccess("done, skotty service installed && started")
					return nil
				}

				s.LogSuccess("done, now you can start skotty agent: %s", startCmd)
				return nil
			})
			if err != nil {
				s.LogError("%s", err)
				s.LogSuccess("done, now you can start skotty agent: skotty start")
			}
			return nil
		}

		setupComplete := func() error {
			if staffKeysUpdated {
				s.LogInfo("Skotty is configured, now you have to configure your SSH client: https://docs.yandex-team.ru/skotty/ssh-client")
				return nil
			}

			s.LogInfo("Skotty is configured, now you have to perform two additional steps:")
			fmt.Println("  1. configure SSH client to use Skotty: https://docs.yandex-team.ru/skotty/ssh-client")
			fmt.Println("  2. add this keys to the Staff:")

			for _, key := range staffKeys {
				fmt.Printf("    - %s:\n%s\n\n", key.Purpose, key.Blob)
			}
			return nil
		}

		keyringSetupComplete := func() error {
			if staffKeysUpdated {
				s.LogInfo("done")
				return nil
			}

			s.LogInfo("done, add this keys to the Staff:")

			for _, key := range staffKeys {
				fmt.Printf("  - %s:\n%s\n\n", key.Purpose, key.Blob)
			}
			return nil
		}

		restartService := func() error {
			s.LogInfo("restart skotty service if any")

			err := osutil.Restart()
			switch {
			case err == nil:
				return nil
			case errors.Is(err, osutil.ErrNotInstalled):
				s.LogWarn("not installed")
				return nil
			case errors.Is(err, osutil.ErrNotSupported):
				s.LogWarn("not supported")
				return nil
			default:
				return err
			}
		}

		type StepFn func() error
		var steps []StepFn
		switch {
		case setupArgs.SocketsOnly:
			steps = []StepFn{
				initializeSockets,
				saveConfig,
				restartService,
			}
		case setupArgs.KeyringOnly:
			steps = []StepFn{
				setupToken,
				saveConfig,
				keyringSetupComplete,
				restartService,
			}
		case setupArgs.ConfigOnly:
			steps = []StepFn{
				saveConfig,
			}
		default:
			steps = []StepFn{
				setupToken,
				initializeSockets,
				saveConfig,
				configureService,
				setupComplete,
			}
		}

		for _, step := range steps {
			if err := step(); err != nil {
				s.LogError("fail: %v", err)
				os.Exit(1)
			}
		}

		return nil
	},
}

func init() {
	flags := setupCmd.PersistentFlags()
	flags.BoolVar(&setupArgs.Reuse, "reuse", false, "reuse previously enrolled Yubikey")
	flags.BoolVar(&setupArgs.AllowAnyKeyring, "any-keyring", false, "try to use all available keyrings (only yubikey used by default)")
	flags.BoolVar(&setupArgs.SocketsOnly, "sockets", false, "setup sockets only")
	flags.BoolVar(&setupArgs.KeyringOnly, "keyring", false, "setup keyring only")
	flags.BoolVar(&setupArgs.ConfigOnly, "config", false, "update config only")
}

func checkTerm() bool {
	msg, ok := osutil.CheckInteractiveTerm()
	if ok {
		return true
	}

	fmt.Println(msg)
	return false
}
