package commands

import (
	"bytes"
	"errors"
	"fmt"
	"log"
	"os"
	"path/filepath"
	"runtime"
	"strings"
	"text/template"

	"github.com/blang/semver/v4"
	"github.com/spf13/cobra"
	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/pubstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/socket"
	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil/sshclient"
)

var sshCmd = &cobra.Command{
	Use:          "ssh",
	SilenceUsage: true,
	Short:        "Various ssh helpers",
}

const sshConfTmpl = `
# Sample of {{.ConfigPath}}
{{.Note}}

{{if .UserConfig}}{{.UserConfig}}{{end}}

Host my-cool.dev.host.yandex.net my-cool1.dev.host.yandex.net
  # Forward default sock to the remote dev machine
  ForwardAgent {{.DefaultSock}}


Host *.yandex.net
  # Forward only "sudo" socket to the untrusted environment
  ForwardAgent {{.SudoSock}}
  # But authenticate through default socket
  IdentityAgent {{.DefaultSock}}
`

var sshConfCmd = &cobra.Command{
	Use:          "conf",
	SilenceUsage: true,
	Short:        "Helps to configure ssh client",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		type SocketConf struct {
			ConfigPath  string
			UserConfig  string
			DefaultSock string
			SudoSock    string
			Note        string
		}

		sshClient := sshclient.BestClient()
		checkVersion := func() error {
			expectedVer, err := semver.Parse("8.2.0")
			if err != nil {
				return fmt.Errorf("shit happens: can't parse expected ssh ver: %w", err)
			}

			actualVer, err := sshClient.Version()
			if err != nil {
				return fmt.Errorf("can't get ssh client version: %w", err)
			}

			if actualVer.LT(expectedVer) {
				msg := "your SSH client (%s) is too old (v8.2+ is required)"
				switch runtime.GOOS {
				case "darwin":
					msg += "; you need to update to macOS Monterey"
				}

				return fmt.Errorf(msg, actualVer.String())
			}

			return nil
		}

		if err := checkVersion(); err != nil {
			return err
		}

		errNotDefault := errors.New("config helper works only with default sockets configuration")
		newConf := func() (*SocketConf, error) {
			defaultSock, err := cfg.Socket(sshClient.SocketName(socket.NameDefault))
			if err != nil {
				return nil, errNotDefault
			}

			sudoSock, err := cfg.Socket(sshClient.SocketName(socket.NameSudo))
			if err != nil {
				return nil, errNotDefault
			}

			configPath, err := sshClient.ConfigPath()
			if err != nil {
				return nil, fmt.Errorf("unable to determine ssh config path: %w", err)
			}

			conf := &SocketConf{
				ConfigPath:  configPath,
				DefaultSock: filepath.ToSlash(defaultSock.Path),
				SudoSock:    filepath.ToSlash(sudoSock.Path),
			}

			if runtime.GOOS == "windows" {
				u, err := osutil.UserName()
				if err != nil {
					return nil, fmt.Errorf("unable to locate current user name: %w", err)
				}
				conf.UserConfig = fmt.Sprintf("User %s", u)
			}

			return conf, nil
		}

		socketConf, err := newConf()
		if err != nil {
			return err
		}

		switch sshClient {
		case sshclient.ClientKindOpenSSH:
			socketConf.Note = "# This configuration is generated for OpenSSH v8.2+"
		case sshclient.ClientKindWin32OpenSSH:
			socketConf.Note = "# This configuration is generated for Win32-OpenSSH v8.6+. If you are using a different client, you must configure it manually: ("
		}

		return template.Must(template.New("ssh_conf").Parse(sshConfTmpl[1:])).Execute(os.Stdout, socketConf)
	},
}

var sshEnvArgs struct {
	Socket string
}

var sshEnvCmd = &cobra.Command{
	Use:          "env",
	SilenceUsage: true,
	Short:        "Prints ssh-agent env",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		socketName := sshEnvArgs.Socket
		if socketName == "" {
			socketName = cfg.SSHAuthSock
		}

		sock, err := cfg.Socket(socketName)
		if err != nil {
			return fmt.Errorf("get socket %q config: %w", socketName, err)
		}

		script := sshutil.SSHAgentScript(sock.Path, skottyctl.AgentPID(cfg.CtlSocketPath))
		_, _ = os.Stdout.Write(script)
		return nil
	},
}

var sshKeysArgs struct {
	Out string
}

var sshKeysCmd = &cobra.Command{
	Use:          "keys",
	SilenceUsage: true,
	Short:        "Prints ssh pub keys",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		kr, err := cfg.NewKeyring()
		if err != nil {
			return fmt.Errorf("can't create keyring: %w", err)
		}
		defer kr.Close()

		var out string
		if sshKeysArgs.Out != "" {
			out, err = filepath.Abs(sshKeysArgs.Out)
			if err != nil {
				return fmt.Errorf("can't prepare output dir: %w", err)
			}

			if err := os.MkdirAll(out, 0o755); err != nil {
				return fmt.Errorf("can't prepare output dir: %w", err)
			}
		}

		pubStore := pubstore.NewFileStore(cfg.SSHKeysPath)
		for _, purpose := range cfg.Keyring.Keys {
			comment := fmt.Sprintf("Skotty key %s on %s", purpose, kr.Name())
			var authorizedKey string
			switch purpose {
			case keyring.KeyPurposeSudo, keyring.KeyPurposeRenew:
				// don't need
			case keyring.KeyPurposeLegacy:
				pubBytes, err := pubStore.ReadKey(kr.Name(), purpose)
				if err != nil {
					return fmt.Errorf("can't read ssh pub for keytype %s: %w", purpose, err)
				}

				authorizedKey = fmt.Sprintf("%s # %s", string(pubBytes), comment)
			default:
				pubBytes, err := pubStore.ReadKey(kr.Name(), purpose)
				if err != nil {
					return fmt.Errorf("can't read ssh pub for key type %s: %w", purpose, err)
				}

				pubKey, _, _, _, err := ssh.ParseAuthorizedKey(pubBytes)
				if err != nil {
					return fmt.Errorf("can't parse ssh pub for key type %s: %s", purpose, err)
				}

				pubCert, ok := pubKey.(*ssh.Certificate)
				if !ok {
					return fmt.Errorf("unexpected ssh pub type for key type %s: %T", purpose, pubKey)
				}

				ca := bytes.TrimSpace(ssh.MarshalAuthorizedKey(pubCert.SignatureKey))
				principals := strings.Join(pubCert.ValidPrincipals, ",")
				authorizedKey = fmt.Sprintf("cert-authority,principals=%q %s # %s", principals, string(ca), comment)
			}

			if authorizedKey == "" {
				continue
			}

			if out != "" {
				filename := filepath.Join(out, fmt.Sprintf("skotty_%s.pub", purpose))
				err := os.WriteFile(filename, []byte(authorizedKey), 0o644)
				if err != nil {
					log.Printf("failed to export %s key: %v\n", purpose, err)
				} else {
					fmt.Printf("pub key for %s exported into: %s\n", purpose, filename)
				}
				continue
			}

			fmt.Printf("- %s:\n%s\n\n", purpose, authorizedKey)
		}
		return nil
	},
}

func init() {
	sshCmd.AddCommand(
		sshConfCmd,
		sshEnvCmd,
		sshKeysCmd,
	)

	sshKeysCmd.PersistentFlags().
		StringVar(&sshKeysArgs.Out, "out", "", "save pub keys into directory")
	sshEnvCmd.PersistentFlags().
		StringVar(&sshEnvArgs.Socket, "socket", "", "socket to use")
}
