package commands

import (
	"context"
	"errors"
	"fmt"
	"os"
	"path/filepath"
	"runtime"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/libs/go/daemon"
	"a.yandex-team.ru/security/skotty/launcher/pkg/launcher"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/skotty/internal/agent"
	"a.yandex-team.ru/security/skotty/skotty/internal/confirm"
	"a.yandex-team.ru/security/skotty/skotty/internal/crash"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/memring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/pubstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/scenario"
	"a.yandex-team.ru/security/skotty/skotty/internal/socket"
	"a.yandex-team.ru/security/skotty/skotty/internal/supervisor"
	"a.yandex-team.ru/security/skotty/skotty/internal/ui"
	"a.yandex-team.ru/security/skotty/skotty/internal/version"
	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil"
)

const (
	shutdownDeadline = 30 * time.Second
	daemonizeTimeout = 10 * time.Second
)

var startArgs struct {
	Daemonize bool
	Tray      bool
	Temporary bool
}

//TODO(buglloc): holly shit!!!
var startCmd = &cobra.Command{
	Use:          "start",
	SilenceUsage: true,
	Short:        "Start skotty agent",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(true)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		if pid := skottyctl.AgentPID(cfg.CtlSocketPath); pid > 0 {
			return fmt.Errorf("skotty already started with pid %d", pid)
		}

		notifyStart := func() error {
			return nil
		}

		if startArgs.Daemonize && startArgs.Temporary {
			return errors.New("--daemonize is not supported with --temporary so far ")
		}

		if startArgs.Daemonize {
			d := daemon.NewDaemon("skotty")
			if d.IsParent() {
				ctx, cancel := context.WithTimeout(context.Background(), daemonizeTimeout)
				defer cancel()

				child, err := d.StartChild(ctx)
				if err != nil {
					return fmt.Errorf("failed to start child process: %w", err)
				}

				fmt.Printf("skotty started with PID: %d\n", child.PID())
				return nil
			}

			notifyStart = func() error {
				return d.NotifyStarted(nil)
			}
		}

		crasherOpts := []crash.Option{
			crash.WithName("agent"),
			crash.WithExecutableFinder(func() (string, error) {
				if !osutil.UnderLauncher() {
					return os.Executable()
				}

				exe, err := launcher.LatestExecutable()
				if err != nil {
					return os.Executable()
				}

				return exe, nil
			}),
		}
		if !startArgs.Daemonize {
			crasherOpts = append(crasherOpts,
				crash.WithStdout(os.Stdout),
				crash.WithStderr(os.Stderr),
			)
		}
		cs := crash.NewSaver(filepath.Dir(cfg.AgentLogPath), crasherOpts...)
		if cs.IsParent() {
			_ = notifyStart()
			exitCode, err := cs.StartChild()
			if err != nil {
				return err
			}

			os.Exit(exitCode)
		}

		logger, err := agent.NewAgentLogger(cfg.AgentLogPath, cfg.LogLevel)
		if err != nil {
			return fmt.Errorf("can't initialize logger: %w", err)
		}

		logger.Info("prepare to start agent",
			log.String("version", version.Full()),
			log.Bool("under_launcher", osutil.UnderLauncher()),
		)
		configureAuthSock := func() error {
			if runtime.GOOS == "windows" {
				return nil
			}

			sock, err := cfg.Socket(cfg.SSHAuthSock)
			if err != nil {
				return err
			}

			exportSocket := func() error {
				if !cfg.Startup.ExportAuthSock {
					return nil
				}

				logger.Info("override env[SSH_AUTH_SOCK]",
					log.String("target_socket", cfg.SSHAuthSock),
					log.String("path", sock.Path))
				return sshutil.ExportAuthSock(sock.Path, os.Getpid())
			}

			overwriteSocket := func() error {
				if !cfg.Startup.ReplaceAuthSock {
					return nil
				}

				logger.Info("replace original SSH_AUTH_SOCK with our",
					log.String("target_socket", cfg.SSHAuthSock),
					log.String("path", sock.Path))

				ok, err := sshutil.ReplaceAuthSock(sock.Path)
				if err != nil {
					return err
				}

				if ok {
					logger.Info("original SSH_AUTH_SOCK replaced")
				}
				return nil
			}

			if err := exportSocket(); err != nil {
				logger.Warn("can't export SSH_AUTH_SOCK", log.Error(err))
			}

			if err := overwriteSocket(); err != nil {
				logger.Warn("can't overwrite original SSH_AUTH_SOCK", log.Error(err))
			}
			return nil
		}

		needRestartErr := errors.New("need restart")
		start := func() error {
			newTemporaryKeyStore := func() (*agent.KeyHolder, error) {
				ps := pubstore.NewMemStore()
				kr := memring.NewMemring()

				s := scenario.NewSetup(
					scenario.WithEnrollUpstream(cfg.EnrollmentService),
					scenario.WithPubStore(ps),
				)

				s.LogInfo("initialize in-memory certs")
				err := s.GenKeys(kr, func(certs []scenario.TokenCert) error {
					return s.Request(kr, func(enrollRsp *skotty.RequestEnrollmentRsp) error {
						s.LogSuccess("to complete the enrollment request, you must to perform two-factor authentication via URL: %s ", enrollRsp.AuthURL)
						return s.OpenInBrowser(enrollRsp.AuthURL, func() error {
							issueReq := scenario.IssueEnrollmentReq{
								EnrollID:  enrollRsp.EnrollmentID,
								AuthToken: enrollRsp.AuthToken,
								Certs:     certs,
							}

							return s.WaitAndIssue(kr, issueReq, func(enrollment *scenario.IssuedEnrollment) error {
								s.LogSuccess("got %d certs", len(enrollment.Certs))
								s.LogWarn("certificates expires at: %s", enrollment.ExpiresAt.Format(time.RFC822))

								return s.UpdateKeys(kr, enrollment.Certs, func(keys ...keyring.KeyPurpose) error {
									availableKeys := make(map[keyring.KeyPurpose]struct{})
									strKeys := make([]string, len(keys))
									for i, k := range keys {
										strKeys[i] = " " + k.String()
										availableKeys[k] = struct{}{}
									}

									s.LogSuccess("setup certificates:%s", strings.Join(strKeys, ","))
									for i := 0; i < len(cfg.Sockets); i++ {
										keys := cfg.Sockets[i].Keys
										last := len(keys)
										for k := 0; k < last; k++ {
											if _, ok := availableKeys[keys[k]]; ok {
												continue
											}
											keys[k] = keys[last-1]
											last--
										}

										cfg.Sockets[i].Keys = keys[:last]
									}
									return nil
								})
							})
						})
					})
				})

				if err != nil {
					return nil, err
				}
				return agent.NewKeyHolder(kr, ps), nil
			}

			newPersistentKeyStore := func() (*agent.KeyHolder, error) {
				kr, err := cfg.NewKeyring()
				if err != nil {
					return nil, fmt.Errorf("can't create keyring: %w", err)
				}

				return agent.NewKeyHolder(kr, pubstore.NewFileStore(cfg.SSHKeysPath)), nil
			}

			var ks *agent.KeyHolder
			if startArgs.Temporary {
				ks, err = newTemporaryKeyStore()
			} else {
				ks, err = newPersistentKeyStore()
			}

			if err != nil {
				return err
			}
			defer ks.Close()

			assetsPath, _ := paths.Assets()
			var appUI ui.App
			if startArgs.Tray {
				trayItems := make([]string, 0, len(cfg.Sockets)+1)
				for _, sock := range cfg.Sockets {
					if sock.Kind == socket.KindDummy {
						continue
					}

					trayItems = append(trayItems, sock.Name)
				}

				appUI, err = ui.NewGraphicalApp(
					ui.WithIconsPath(filepath.Join(assetsPath, "icons")),
					ui.WithLogger(logger.WithName("ui")),
					ui.WithSocketsNames(trayItems...),
				)
			} else {
				appUI, err = ui.NewConsoleApp(
					ui.WithIconsPath(filepath.Join(assetsPath, "icons")),
					ui.WithLogger(logger.WithName("ui")),
				)
			}

			if err != nil {
				return fmt.Errorf("can't initialize UI: %w", err)
			}
			defer func() {
				ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
				defer cancel()
				if err := appUI.Shutdown(ctx); err != nil {
					logger.Error("failed to shutdown ui", log.Error(err))
				}
			}()

			confirmator, err := confirm.NewConfirmator(cfg.Confirm.Kind, cfg.Confirm.Program)
			if err != nil {
				return fmt.Errorf("failed to create confimator: %w", err)
			}

			agents, err := agent.NewAgentsGang(ks,
				agent.WithLogger(logger),
				agent.WithNotifier(appUI),
				agent.WithSockets(cfg.Sockets...),
				agent.WithConfirmator(confirmator),
			)
			if err != nil {
				return fmt.Errorf("failed to create agents gang: %w", err)
			}

			doRestart := make(chan struct{})
			ctl := supervisor.NewSupervisor(
				cfg.CtlSocketPath,
				supervisor.WithLogger(logger),
				supervisor.WithNotifier(appUI),
				supervisor.WithCertsChecker(ks.PubStore, ks.Keyring.Name(), ks.Keyring.SupportedKeyTypes()...),
				supervisor.WithKeyReloadHandler(func() error {
					return agents.ReloadKeys()
				}),
				supervisor.WithRestartHandler(func() error {
					close(doRestart)
					return nil
				}),
			)

			return appUI.Start(func(ctx context.Context) error {
				agentErr := make(chan error, 1)
				go func() {
					defer close(agentErr)

					logger.Info("agent starting")
					if err := configureAuthSock(); err != nil {
						logger.Error("can't override SSH_AUTH_SOCK", log.Error(err))
					}

					if err := agents.ListenAndServe(); err != nil {
						logger.Error("failed to start agent", log.Error(err))
						agentErr <- err
					}
					logger.Info("agent stopped")
				}()

				go func() {
					logger.Info("supervisor starting")
					if err := ctl.ListenAndServe(); err != nil {
						logger.Error("failed to start supervisor", log.Error(err))
					}
					logger.Info("supervisor stopped")
				}()

				shutdown := func() {
					ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
					defer cancel()

					if err := agents.Shutdown(ctx); err != nil {
						logger.Error("failed to shutdown agent", log.Error(err))
					}

					if err := ctl.Shutdown(ctx); err != nil {
						logger.Error("failed to shutdown ctl", log.Error(err))
					}
				}

				defer func() {
					fmt.Println("agent stopped")
				}()

				fmt.Printf("agent started with log: %s\n", cfg.AgentLogPath)
				select {
				case <-doRestart:
					logger.Info("restart gracefully")
					fmt.Println("restart gracefully")

					shutdown()
					return needRestartErr

				case <-ctx.Done():
					logger.Info("shutting down gracefully")
					fmt.Println("shutting down gracefully")

					shutdown()
					return nil

				case err := <-agentErr:
					return err
				}
			})
		}

		if err := start(); err != nil {
			if errors.Is(err, needRestartErr) {
				os.Exit(crash.NeedRestartExitCode)
				return nil
			}

			logger.Error("agent failed to start", log.Error(err))
			return err
		}

		return nil
	},
}

func init() {
	flags := startCmd.PersistentFlags()
	flags.BoolVar(&startArgs.Daemonize, "daemonize", false, "daemonize skotty and print info about it (like ssh-agent)")
	flags.BoolVar(&startArgs.Tray, "tray", false, "show tray icon")
	flags.BoolVar(&startArgs.Temporary, "temporary", false, "starts with temporary (in-memory) certs storage")
}
