package commands

import (
	"encoding/base64"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/libs/go/iokit"
	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/logger"
	"a.yandex-team.ru/security/skotty/skotty/internal/passutil"
	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/asker"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/scenario"
	"a.yandex-team.ru/security/skotty/skotty/internal/yubiconf"
	"a.yandex-team.ru/security/skotty/skotty/internal/yubikey"
	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
)

var yubikeyCmd = &cobra.Command{
	Use:          "yubikey",
	SilenceUsage: true,
	Short:        "various Yubikey helpers",
}

var yubikeyListCmd = &cobra.Command{
	Use:          "list",
	SilenceUsage: true,
	Short:        "list available Yubikeys",
	RunE: func(_ *cobra.Command, _ []string) error {
		cards, err := yubikey.Cards()
		if err != nil {
			return fmt.Errorf("can't list yubikeys: %w", err)
		}

		cfg, _ := loadConfig(false)
		for _, card := range cards {
			name := card.Name
			if cfg != nil && card.Serial == cfg.Keyring.Yubikey.Serial {
				name += " (used by skotty)"
			}

			func() {
				fmt.Println(name)
				fmt.Printf("  - serial: %d\n", card.Serial)
				yk, err := yubikey.Open(card)
				if err != nil {
					logger.Error("failed to open yubikey", log.UInt32("serial", card.Serial), log.Error(err))
					return
				}
				defer yk.Close()

				fmt.Printf("  - version: %s\n", yk.Version())
				fmt.Println("  - keys:")
				keys, err := yk.ListKeys(yubikey.AllSlots...)
				if err != nil {
					logger.Error("failed to list keys", log.UInt32("serial", card.Serial), log.Error(err))
					return
				}

				for _, key := range keys {
					fmt.Printf("    * %s %s:\n      - serial: %s\n      - expires: %s\n",
						key.Slot, key.Subject.CommonName,
						key.SerialNumber.String(), key.NotAfter.Local().Format(time.RFC822))
				}
			}()
		}

		return nil
	},
}

var yubikeyUnblockCmd = &cobra.Command{
	Use:          "unblock",
	SilenceUsage: true,
	Short:        "unblock yubikey with PUK",
	RunE: func(_ *cobra.Command, _ []string) error {
		cards, err := yubikey.Cards()
		if err != nil {
			return fmt.Errorf("can't list yubikeys: %w", err)
		}

		card, err := asker.SelectYubikey(cards...)
		if err != nil {
			return err
		}

		newPIN, err := passutil.Password(6)
		if err != nil {
			return fmt.Errorf("can't generate new PIN: %w", err)
		}

		var message string
		for {
			puk, err := asker.AskPUK(message)
			if err != nil {
				return err
			}

			yk, err := yubikey.Open(card)
			if err != nil {
				return fmt.Errorf("can't open yubikey %q: %w", card, err)
			}

			err = yk.Unblock(puk, newPIN)
			if err != nil {
				message = fmt.Sprintf("Auth failed: %s", err)
			}

			yk.Close()
			if err == nil {
				break
			}
		}

		fmt.Printf("Unblocked. New PIN: %s\n", newPIN)

		cfg, err := loadConfig(true)
		if err != nil || cfg.Keyring.Yubikey.Serial != card.Serial {
			fmt.Println("Config not updated due to uses another Yubikey")
			fmt.Println("Done")
			return nil
		}

		err = func() error {
			cfgPath, err := paths.Config()
			if err != nil {
				return err
			}

			if cfg.Keyring.Yubikey.PIN.CanStore() {
				kr, err := cfg.NewKeyring()
				if err != nil {
					return fmt.Errorf("unable to create keyring: %w", err)
				}

				if err := cfg.Keyring.Yubikey.PIN.StorePIN(newPIN, kr.PinStoreOpts()...); err != nil {
					return fmt.Errorf("unable to store PIN: %w", err)
				}
			}

			if err := config.Save(cfg, cfgPath); err != nil {
				return fmt.Errorf("can't update config: %w", err)
			}

			fmt.Printf("Config updated: %s\n", cfgPath)
			return nil
		}()

		if err != nil {
			return err
		}

		err = osutil.Restart()
		if err != nil {
			fmt.Printf("Failed to restart skotty: %s\n", err)
			fmt.Println("You must to restart it manually")
		} else {
			fmt.Println("Skotty restarted")
		}

		fmt.Println("Done")
		return nil
	},
}

var yubikeyChangePINCmd = &cobra.Command{
	Use:          "change-pin",
	SilenceUsage: true,
	Short:        "change yubikey PIN",
	RunE: func(_ *cobra.Command, _ []string) error {
		cfg, err := loadConfig(false)
		if err != nil {
			return fmt.Errorf("failed to load config: %w", err)
		}

		cards, err := yubikey.Cards()
		if err != nil {
			return fmt.Errorf("can't list yubikeys: %w", err)
		}

		card, err := asker.SelectYubikey(cards...)
		if err != nil {
			return err
		}

		checkAuth := func(pin string) error {
			yk, err := yubikey.Open(card)
			if err != nil {
				return err
			}

			_, err = yk.MngmtKey(pin)
			yk.Close()
			return err
		}

		confPin := func() string {
			if cfg == nil {
				return ""
			}

			if cfg.Keyring.Kind != keyring.KindYubikey {
				return ""
			}

			if card.Serial != cfg.Keyring.Yubikey.Serial {
				return ""
			}

			serial := strconv.Itoa(int(cfg.Keyring.Yubikey.Serial))
			pin, err := cfg.Keyring.Yubikey.PIN.GetPIN(
				func(_ string) error {
					return nil
				},
				pinstore.WithSerial(serial),
				pinstore.WithDescription("Please enter the PIN to unlock Yubikey #"+serial),
			)
			if err != nil {
				return ""
			}

			return pin
		}()

		pin := confPin
		for {
			var message string
			if pin != "" {
				err := checkAuth(pin)
				if err == nil {
					break
				}

				message = fmt.Sprintf("Auth failed: %s", err)
			}

			pin, err = asker.AskYubiPIN(message)
			if err != nil {
				return err
			}
		}

		newPIN, err := asker.AskNewPIN()
		if err != nil {
			return err
		}

		yk, err := yubikey.Open(card)
		if err != nil {
			return fmt.Errorf("can't open yubikey: %w", err)
		}

		err = yk.SetPIN(pin, newPIN)
		yk.Close()
		if err != nil {
			return fmt.Errorf("can't setup new PIN: %w", err)
		}

		fmt.Println("PIN updated")
		if cfg == nil || cfg.Keyring.Yubikey.Serial != card.Serial {
			fmt.Println("Config not updated due to uses another Yubikey")
			fmt.Println("Done")
			return nil
		}

		cfgPath, err := paths.Config()
		if err != nil {
			return err
		}

		if cfg.Keyring.Yubikey.PIN.CanStore() {
			kr, err := cfg.NewKeyring()
			if err != nil {
				return fmt.Errorf("unable to create keyring: %w", err)
			}

			if err := cfg.Keyring.Yubikey.PIN.StorePIN(newPIN, kr.PinStoreOpts()...); err != nil {
				return fmt.Errorf("unable to store PIN: %w", err)
			}
		}

		if err := config.Save(cfg, cfgPath); err != nil {
			return fmt.Errorf("can't update config: %w", err)
		}

		fmt.Printf("Config updated: %s\n", cfgPath)

		err = osutil.Restart()
		if err != nil {
			fmt.Printf("Failed to restart skotty: %s\n", err)
			fmt.Println("You must to restart it manually")
		} else {
			fmt.Println("Skotty restarted")
		}

		fmt.Println("Done")
		return nil
	},
}

var yubikeyResetCmd = &cobra.Command{
	Use:          "reset",
	SilenceUsage: true,
	Short:        "reset yubikey",
	RunE: func(_ *cobra.Command, _ []string) error {
		cards, err := yubikey.Cards()
		if err != nil {
			return fmt.Errorf("can't list yubikeys: %w", err)
		}

		card, err := asker.SelectYubikey(cards...)
		if err != nil {
			return err
		}

		userPIN, err := asker.ConfirmProvidePIN()
		if err != nil {
			return err
		}

		var pin, puk string
		if userPIN {
			pin, err = asker.AskNewPIN()
			if err != nil {
				return err
			}

			puk, err = asker.AskNewPUK()
			if err != nil {
				return err
			}
		} else {
			pin, err = passutil.Password(6)
			if err != nil {
				return err
			}

			puk, err = passutil.Password(8)
			if err != nil {
				return err
			}
		}

		yk, err := yubikey.Open(card)
		if err != nil {
			return fmt.Errorf("failed to open yubikey: %w", err)
		}

		defer yk.Close()
		if ok, _ := asker.ConfirmReset(card); !ok {
			fmt.Println("aborted")
			return nil
		}

		if err = yk.Reset(pin, puk); err != nil {
			return fmt.Errorf("reset failed: %w", err)
		}

		if userPIN {
			fmt.Printf("New PIN: %s\n", passutil.Mask(pin))
			fmt.Printf("New PUK: %s\n", passutil.Mask(puk))
		} else {
			fmt.Printf("New PIN: %s\n", pin)
			fmt.Printf("New PUK: %s\n", puk)
		}

		return nil
	},
}

var yubikeyUSBCmdAgrs = struct {
	Enable  []string
	Disable []string
}{}

var yubikeyUSBCmd = &cobra.Command{
	Use:          "usb",
	SilenceUsage: true,
	Short:        "manage USB config",
	RunE: func(_ *cobra.Command, _ []string) error {
		if len(yubikeyUSBCmdAgrs.Enable) == 0 && len(yubikeyUSBCmdAgrs.Disable) == 0 {
			return errors.New("you must provide --enable or --disable flag")
		}

		cards, err := yubikey.Cards()
		if err != nil {
			return fmt.Errorf("can't list yubikeys: %w", err)
		}

		card, err := asker.SelectYubikey(cards...)
		if err != nil {
			return err
		}

		yc, err := yubiconf.Open(card)
		if err != nil {
			return err
		}
		defer yc.Close()

		for _, app := range yubikeyUSBCmdAgrs.Enable {
			appID, err := yubiconf.ParseApplicationID(app)
			if err != nil {
				return err
			}

			err = yc.EnableUSBApp(appID)
			if err != nil {
				return fmt.Errorf("failed to enable USB app %q: %w", app, err)
			}
		}

		for _, app := range yubikeyUSBCmdAgrs.Disable {
			appID, err := yubiconf.ParseApplicationID(app)
			if err != nil {
				return err
			}

			err = yc.DisableUSBApp(appID)
			if err != nil {
				return fmt.Errorf("failed to disable USB app %q: %w", app, err)
			}
		}

		if err := yc.WriteConfig(true); err != nil {
			return fmt.Errorf("failed to write yubikey config: %w", err)
		}

		if len(yubikeyUSBCmdAgrs.Enable) > 0 {
			fmt.Println("Enabled apps:")
			for _, app := range yubikeyUSBCmdAgrs.Enable {
				fmt.Printf("  - %s\n", app)
			}
		}

		if len(yubikeyUSBCmdAgrs.Disable) > 0 {
			fmt.Println("Disabled apps:")
			for _, app := range yubikeyUSBCmdAgrs.Disable {
				fmt.Printf("  - %s\n", app)
			}
		}

		// we need to restart skotty to use correct smart-card "reader name"
		err = osutil.Restart()
		if err != nil {
			fmt.Printf("Failed to restart skotty: %s\n", err)
			fmt.Println("You must to restart it manually")
		} else {
			fmt.Println("Skotty restarted")
		}

		return nil
	},
}

var yubikeyDumpCmd = &cobra.Command{
	Use:          "dump",
	SilenceUsage: true,
	Short:        "dumps various yubikey info (object and so on)",
}

var yubikeyDumpObjectCmd = &cobra.Command{
	Use:          "object",
	SilenceUsage: true,
	Short:        "dump yubikey object (objects list: https://developers.yubico.com/yubico-piv-tool/Actions/read_write_objects.html)",
	RunE: func(_ *cobra.Command, args []string) error {
		cards, err := yubikey.Cards()
		if err != nil {
			return fmt.Errorf("can't list yubikeys: %w", err)
		}

		card, err := asker.SelectYubikey(cards...)
		if err != nil {
			return err
		}

		pc, err := yubiconf.NewPCSCClient(card)
		if err != nil {
			return err
		}
		defer pc.Close()

		confPin := func() string {
			cfg, err := loadConfig(false)
			if err != nil {
				return ""
			}

			if cfg == nil {
				return ""
			}

			if cfg.Keyring.Kind != keyring.KindYubikey {
				return ""
			}

			if card.Serial != cfg.Keyring.Yubikey.Serial {
				return ""
			}

			serial := strconv.Itoa(int(cfg.Keyring.Yubikey.Serial))
			pin, err := cfg.Keyring.Yubikey.PIN.GetPIN(
				func(_ string) error {
					return nil
				},
				pinstore.WithSerial(serial),
				pinstore.WithDescription("Please enter the PIN to unlock Yubikey #"+serial),
			)
			if err != nil {
				return ""
			}

			return pin
		}()

		pin := confPin
		for {
			var message string
			if pin != "" {
				err := pc.Login(pin)
				if err == nil {
					break
				}

				message = fmt.Sprintf("Auth failed: %s", err)
			}

			pin, err = asker.AskYubiPIN(message)
			if err != nil {
				return err
			}
		}

		for _, rawID := range args {
			id, err := strconv.ParseUint(strings.TrimPrefix(rawID, "0x"), 16, 32)
			if err != nil {
				return fmt.Errorf("invalid object id %q: %w", rawID, err)
			}

			data, err := pc.ReadObject(uint32(id))
			var out string
			if err != nil {
				out = err.Error()
			} else {
				out = base64.StdEncoding.EncodeToString(data)
			}

			fmt.Printf("%s: %s\n", rawID, out)
		}

		return nil
	},
}

var yubikeyRebootCmd = &cobra.Command{
	Use:          "reboot",
	SilenceUsage: true,
	Short:        "reboot Yubikey",
	RunE: func(_ *cobra.Command, args []string) error {
		cards, err := yubikey.Cards()
		if err != nil {
			return fmt.Errorf("can't list yubikeys: %w", err)
		}

		card, err := asker.SelectYubikey(cards...)
		if err != nil {
			return err
		}

		yc, err := yubiconf.Open(card)
		if err != nil {
			return fmt.Errorf("can't open yubikey %q: %w", card, err)
		}
		defer yc.Close()

		if err := yc.Reboot(); err != nil {
			return fmt.Errorf("unable to reboot yubieky %q: %w", card, err)
		}

		fmt.Println("reboot request has been sent, wait for Yubikey to bring back...")
		wait := func() error {
			check := func() (bool, error) {
				cards, err := yubikey.Cards()
				if err != nil {
					return false, fmt.Errorf("can't listen yubikeys: %w", err)
				}

				for _, candidate := range cards {
					if candidate.Serial == card.Serial {
						return true, nil
					}
				}

				return false, nil
			}

			for {
				time.Sleep(500 * time.Millisecond)
				exists, err := check()
				if err != nil {
					return err
				}

				if exists {
					return nil
				}
			}
		}

		if err := wait(); err != nil {
			return err
		}

		fmt.Println("done")
		return nil
	},
}

var yubikeyReplugArgs struct {
	VendorID  int
	ProductID int
}

var yubikeyReplugCmd = &cobra.Command{
	Use:          "replug",
	SilenceUsage: true,
	Short:        "replug Yubikey USB device",
	RunE: func(_ *cobra.Command, args []string) error {
		s := scenario.NewReplug()

		return s.ListUSBDevices(yubikeyReplugArgs.VendorID, yubikeyReplugArgs.ProductID, func(devices []iokit.Device) error {
			if err := s.PrintUSBDevices(devices); err != nil {
				return err
			}

			if len(devices) == 0 {
				return errors.New("no suitable device found")
			}

			err := s.ListPIVDevices(s.PrintPIVCards)
			if err != nil {
				return err
			}

			err = s.ReplugSmartCard(devices[0], func() error {
				time.Sleep(time.Second)
				s.LogSuccess("Yubikey and smart card subsystem must be respawned")
				return nil
			})
			if err != nil {
				return err
			}

			return s.ListPIVDevices(s.PrintPIVCards)
		})
	},
}

func init() {
	var apps strings.Builder
	for i, app := range yubiconf.AllApplications {
		if i != 0 {
			apps.WriteString(", ")
		}
		apps.WriteString(app.String())
	}

	usbFlags := yubikeyUSBCmd.PersistentFlags()
	usbFlags.StringSliceVar(&yubikeyUSBCmdAgrs.Enable, "enable", nil, fmt.Sprintf("enable applications (%s)", apps.String()))
	usbFlags.StringSliceVar(&yubikeyUSBCmdAgrs.Disable, "disable", nil, fmt.Sprintf("disable applications (%s)", apps.String()))

	replugFlags := yubikeyReplugCmd.PersistentFlags()
	replugFlags.IntVar(&yubikeyReplugArgs.VendorID, "vendor-id", 0x1050, "USB vendor id")
	replugFlags.IntVar(&yubikeyReplugArgs.ProductID, "product-id", 0, "USB product id, full list: https://support.yubico.com/hc/en-us/articles/360016614920-YubiKey-USB-ID-Values")

	yubikeyDumpCmd.AddCommand(
		yubikeyDumpObjectCmd,
	)

	yubikeyCmd.AddCommand(
		yubikeyListCmd,
		yubikeyChangePINCmd,
		yubikeyUnblockCmd,
		yubikeyResetCmd,
		yubikeyUSBCmd,
		yubikeyDumpCmd,
		yubikeyRebootCmd,
		yubikeyReplugCmd,
	)
}
