package config

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*ConfirmKind)(nil)
var _ yaml.Marshaler = (*ConfirmKind)(nil)
var _ json.Unmarshaler = (*ConfirmKind)(nil)
var _ json.Marshaler = (*ConfirmKind)(nil)

type ConfirmKind uint8

const (
	ConfirmKindNone ConfirmKind = iota
	ConfirmKindAny
	ConfirmKindAdded
	ConfirmKindKeyring
)

func (k ConfirmKind) String() string {
	switch k {
	case ConfirmKindNone:
		return ""
	case ConfirmKindAny:
		return "any"
	case ConfirmKindAdded:
		return "added"
	case ConfirmKindKeyring:
		return "keyring"
	default:
		return fmt.Sprintf("unknown_%d", k)
	}
}

func (k *ConfirmKind) fromString(s string) error {
	switch s {
	case "":
		*k = ConfirmKindNone
	case "any":
		*k = ConfirmKindAny
	case "added":
		*k = ConfirmKindAdded
	case "keyring":
		*k = ConfirmKindKeyring
	default:
		return fmt.Errorf("unknown confirm kind: %s", s)
	}
	return nil
}

func (k ConfirmKind) MarshalYAML() (interface{}, error) {
	return k.String(), nil
}

func (k *ConfirmKind) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return k.fromString(s)
}

func (k ConfirmKind) MarshalJSON() ([]byte, error) {
	return json.Marshal(k.String())
}

func (k *ConfirmKind) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return k.fromString(s)
}
