package confirm

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*Kind)(nil)
var _ yaml.Marshaler = (*Kind)(nil)
var _ json.Unmarshaler = (*Kind)(nil)
var _ json.Marshaler = (*Kind)(nil)

type Kind uint8

const (
	KindNone Kind = iota
	KindSSHAskPass
	KindPinentry
)

func (k Kind) String() string {
	switch k {
	case KindNone:
		return ""
	case KindSSHAskPass:
		return "ssh-askpass"
	case KindPinentry:
		return "pinentry"
	default:
		return fmt.Sprintf("unknown_%d", k)
	}
}

func (k *Kind) fromString(s string) error {
	switch s {
	case "":
		*k = KindNone
	case "ssh-askpass":
		*k = KindSSHAskPass
	case "pinentry":
		*k = KindPinentry
	default:
		return fmt.Errorf("unknown confirmator: %s", s)
	}
	return nil
}

func (k Kind) MarshalYAML() (interface{}, error) {
	return k.String(), nil
}

func (k *Kind) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return k.fromString(s)
}

func (k Kind) MarshalJSON() ([]byte, error) {
	return json.Marshal(k.String())
}

func (k *Kind) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return k.fromString(s)
}
