package confirm

import (
	"errors"
	"fmt"
	"sync"

	"golang.org/x/sys/execabs"

	"a.yandex-team.ru/security/skotty/libs/pinentry"
)

var _ Confirmator = (*Pinentry)(nil)

type Pinentry struct {
	program string
	mu      sync.Mutex
}

func NewPinentry(program string) *Pinentry {
	return &Pinentry{
		program: program,
	}
}

func (p *Pinentry) Confirm(msg string) (bool, error) {
	p.mu.Lock()
	defer p.mu.Unlock()

	if err := p.initProgram(); err != nil {
		return false, err
	}

	client, err := pinentry.NewClient(
		pinentry.WithBinaryName(p.program),
		pinentry.WithGPGTTY(),
		pinentry.WithTitle("Skotty SSH"),
		pinentry.WithPrompt("Skotty SSH"),
		pinentry.WithDesc(msg),
	)
	if err != nil {
		return false, fmt.Errorf("unable to create pinentry client: %w", err)
	}
	defer func() { _ = client.Close() }()

	return client.Confirm("")
}

func (p *Pinentry) initProgram() error {
	if p.program != "" {
		return nil
	}

	for _, candidate := range pinentry.DefaultProgs {
		if bin, err := execabs.LookPath(candidate); err == nil {
			p.program = bin
			return nil
		}
	}

	return errors.New("unable to find suitable pinentry program, your must to provide in manually")
}
