package executil

import (
	"os"
	"path/filepath"
	"strings"
)

func SuggestExecutables(name string) ([]string, error) {
	exts := make(map[string]struct{})
	x := os.Getenv(`PATHEXT`)
	if x != "" {
		for _, e := range strings.Split(strings.ToLower(x), `;`) {
			if e == "" {
				continue
			}
			if e[0] != '.' {
				e = "." + e
			}
			exts[e] = struct{}{}
		}
	} else {
		exts = map[string]struct{}{
			".exe": {},
			".bat": {},
		}
	}

	if strings.ContainsAny(name, `:\/`) {
		var globName string
		if stat, err := os.Stat(name); err == nil && stat.IsDir() {
			globName = filepath.Join(name, "*")
		} else {
			globName = name + "*"
		}

		files, err := filepath.Glob(globName)
		if err != nil {
			return nil, err
		}

		out := make([]string, 0, len(files))
		for _, f := range files {
			d, err := os.Stat(f)
			if err != nil {
				continue
			}

			if _, ok := exts[filepath.Ext(f)]; ok {
				out = append(out, f)
				continue
			}

			if m := d.Mode(); m.IsDir() {
				out = append(out, f)
			}
		}
		return out, nil
	}

	globName := name + "*"
	path := os.Getenv("path")
	var out []string
	for _, dir := range filepath.SplitList(path) {
		files, err := filepath.Glob(filepath.Join(dir, globName))
		if err != nil {
			continue
		}

		for _, f := range files {
			if _, ok := exts[filepath.Ext(f)]; ok {
				out = append(out, f)
				continue
			}
		}
	}

	return out, nil
}
