package keyring

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*Algorithm)(nil)
var _ yaml.Marshaler = (*Algorithm)(nil)
var _ json.Unmarshaler = (*Algorithm)(nil)
var _ json.Marshaler = (*Algorithm)(nil)

type Algorithm int

const (
	AlgorithmNone Algorithm = iota + 1
	AlgorithmEC256
	AlgorithmRSA1024
)

func (a Algorithm) String() string {
	switch a {
	case AlgorithmEC256:
		return "ec256"
	case AlgorithmRSA1024:
		return "rsa1024"
	case AlgorithmNone:
		return "none"
	default:
		return fmt.Sprintf("algo_%d", a)
	}
}

func (a *Algorithm) fromString(v string) error {
	switch v {
	case "ec256":
		*a = AlgorithmEC256
	case "rsa1024":
		*a = AlgorithmRSA1024
	case "none":
		*a = AlgorithmNone
	default:
		return fmt.Errorf("unknown key algo: %s", v)
	}

	return nil
}

func (a Algorithm) MarshalYAML() (interface{}, error) {
	return a.String(), nil
}

func (a *Algorithm) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return a.fromString(s)
}

func (a Algorithm) MarshalJSON() ([]byte, error) {
	return json.Marshal(a.String())
}

func (a *Algorithm) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return a.fromString(s)
}
