package keyring

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*KeyPurpose)(nil)
var _ yaml.Marshaler = (*KeyPurpose)(nil)
var _ json.Unmarshaler = (*KeyPurpose)(nil)
var _ json.Marshaler = (*KeyPurpose)(nil)

type KeyPurpose uint8

const (
	KeyPurposeNone KeyPurpose = iota
	KeyPurposeSudo
	KeyPurposeSecure
	KeyPurposeInsecure
	KeyPurposeLegacy
	KeyPurposeRenew
)

func (k KeyPurpose) String() string {
	switch k {
	case KeyPurposeNone:
		return "none"
	case KeyPurposeSecure:
		return "secure"
	case KeyPurposeInsecure:
		return "insecure"
	case KeyPurposeSudo:
		return "sudo"
	case KeyPurposeLegacy:
		return "legacy"
	case KeyPurposeRenew:
		return "renew"
	default:
		return fmt.Sprintf("key_%d", k)
	}
}

func (k *KeyPurpose) fromString(v string) error {
	switch v {
	case "none":
		*k = KeyPurposeNone
	case "secure":
		*k = KeyPurposeSecure
	case "insecure":
		*k = KeyPurposeInsecure
	case "sudo":
		*k = KeyPurposeSudo
	case "renew":
		*k = KeyPurposeRenew
	case "legacy":
		*k = KeyPurposeLegacy
	default:
		return fmt.Errorf("unknown key purpose: %s", v)
	}

	return nil
}

func (k KeyPurpose) MarshalYAML() (interface{}, error) {
	return k.String(), nil
}

func (k *KeyPurpose) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return k.fromString(s)
}

func (k KeyPurpose) MarshalJSON() ([]byte, error) {
	return json.Marshal(k.String())
}

func (k *KeyPurpose) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return k.fromString(s)
}

func (k KeyPurpose) Algo() Algorithm {
	switch k {
	case KeyPurposeLegacy:
		return AlgorithmRSA1024
	default:
		return AlgorithmEC256
	}
}

func (k KeyPurpose) TouchPolicy() TouchPolicy {
	switch k {
	case KeyPurposeSecure:
		return TouchPolicyCached
	case KeyPurposeInsecure:
		return TouchPolicyNever
	case KeyPurposeSudo:
		return TouchPolicyAlways
	case KeyPurposeLegacy:
		return TouchPolicyNever
	case KeyPurposeRenew:
		return TouchPolicyNever
	default:
		return TouchPolicyAlways
	}
}

func ContainsKeyPurpose(s []KeyPurpose, purpose KeyPurpose) bool {
	for _, keyType := range s {
		if keyType == purpose {
			return true
		}
	}

	return false
}
