//go:build darwin
// +build darwin

package keychain

/*
#cgo LDFLAGS: -framework CoreFoundation -framework Security

#include <CoreFoundation/CoreFoundation.h>
#include <Security/Security.h>
*/
import "C"
import (
	"encoding/json"
	"errors"
	"fmt"

	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/keychain/internal/chaintypes"
	"a.yandex-team.ru/security/skotty/skotty/internal/osxkeychain"
)

const (
	keychainService = "skotty"
)

var _ chaintypes.Service = (*Service)(nil)
var _ chaintypes.Session = (*Session)(nil)

type Service struct {
	collection string
}

type Session struct {
	collection string
}

func NewService(collection string) (*Service, error) {
	return &Service{
		collection: collection,
	}, nil
}

func (s *Service) Session() (chaintypes.Session, error) {
	return &Session{
		collection: s.collection,
	}, nil
}

func (s *Service) Close() {}

func (t *Session) FetchKeyPair(keyType keyring.KeyPurpose) (chaintypes.KeyPair, error) {
	var out chaintypes.KeyPair
	query := osxkeychain.NewItem()
	query.SetSecClass(osxkeychain.SecClassGenericPassword)
	query.SetService(keychainService)
	query.SetAccount(t.keyAccount(keyType))
	query.SetLabel(t.keyLabel(keyType))
	query.SetMatchLimit(osxkeychain.MatchLimitOne)
	query.SetReturnData(true)
	results, err := osxkeychain.QueryItem(query)
	if err != nil {
		return out, err
	}

	if len(results) > 1 {
		return out, fmt.Errorf("too many results: %d", len(results))
	}

	if len(results) == 0 {
		return out, fmt.Errorf("item not found for label: %s", t.keyLabel(keyType))
	}

	err = json.Unmarshal(results[0].Data, &out)
	if err != nil {
		return out, fmt.Errorf("failed to unmarshal keypair: %w", err)
	}

	return out, nil
}

func (t *Session) SaveKeyPair(keyType keyring.KeyPurpose, keyPair chaintypes.KeyPair) error {
	secBlob, err := json.Marshal(keyPair)
	if err != nil {
		return fmt.Errorf("failed to marshal certificate: %w", err)
	}

	item := osxkeychain.NewItem()
	item.SetSecClass(osxkeychain.SecClassGenericPassword)
	item.SetService(keychainService)
	item.SetAccount(t.keyAccount(keyType))
	item.SetLabel(t.keyLabel(keyType))
	item.SetAccessible(osxkeychain.AccessibleWhenUnlocked)
	item.SetData(secBlob)
	err = osxkeychain.AddItem(item)
	if err != nil {
		if errors.Is(err, osxkeychain.ErrorDuplicateItem) {
			item.SetMatchLimit(osxkeychain.MatchLimitOne)
			item.SetData(nil)

			update := osxkeychain.NewItem()
			update.SetData(secBlob)
			update.SetAccessible(osxkeychain.AccessibleWhenUnlocked)
			return osxkeychain.UpdateItem(item, update)
		}
		return err
	}

	return nil
}

func (t *Session) Close() {}

func (t *Session) keyAccount(keyType keyring.KeyPurpose) string {
	return fmt.Sprintf("%s@skotty", keyType)
}

func (t *Session) keyLabel(keyType keyring.KeyPurpose) string {
	a := t.keyAccount(keyType)
	if t.collection == "" {
		return a
	}
	return fmt.Sprintf("%s@%s", a, t.collection)
}
