//go:build linux
// +build linux

package secretservice

import (
	"context"
	"sync"

	"github.com/godbus/dbus/v5"
)

type SignalDispatcher struct {
	dbusConn  *dbus.Conn
	signals   chan *dbus.Signal
	handlers  []*SignalHandler
	mu        sync.Mutex
	ctx       context.Context
	cancelCtx context.CancelFunc
}

type SignalHandler struct {
	name string
	path dbus.ObjectPath
	res  chan []interface{}
}

func NewSignalDispatcher(dbusConn *dbus.Conn) *SignalDispatcher {
	ctx, cancelCtx := context.WithCancel(context.Background())

	signals := make(chan *dbus.Signal, 16)
	dbusConn.Signal(signals)

	sd := &SignalDispatcher{
		dbusConn:  dbusConn,
		signals:   signals,
		ctx:       ctx,
		cancelCtx: cancelCtx,
	}

	go sd.loop()
	return sd
}

func (s *SignalDispatcher) WaitEvent(signalName string, targetPath dbus.ObjectPath) []interface{} {
	h := SignalHandler{
		name: signalName,
		path: targetPath,
		res:  make(chan []interface{}, 1),
	}

	s.mu.Lock()
	s.handlers = append(s.handlers, &h)
	s.mu.Unlock()

	out, ok := <-h.res
	if ok {
		close(h.res)
	}
	return out
}

func (s *SignalDispatcher) Close() {
	s.cancelCtx()

	s.mu.Lock()
	for _, h := range s.handlers {
		close(h.res)
	}
	s.mu.Unlock()
}

func (s *SignalDispatcher) loop() {
	for {
		select {
		case signal := <-s.signals:
			s.mu.Lock()
			tmp := s.handlers[:0]
			for _, h := range s.handlers {
				if signal.Name != h.name || signal.Path != h.path {
					tmp = append(tmp, h)
					continue
				}

				h.res <- signal.Body
			}
			s.handlers = tmp
			s.mu.Unlock()
		case <-s.ctx.Done():
			return
		}
	}
}
