package keyring

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*Kind)(nil)
var _ yaml.Marshaler = (*Kind)(nil)
var _ json.Unmarshaler = (*Kind)(nil)
var _ json.Marshaler = (*Kind)(nil)

type Kind uint8

const (
	KindNone Kind = iota
	KindYubikey
	KindKeychain
	KindFiles
)

func (k Kind) String() string {
	switch k {
	case KindYubikey:
		return "yubikey"
	case KindKeychain:
		return "keychain"
	case KindFiles:
		return "files"
	case KindNone:
		return "none"
	default:
		return fmt.Sprintf("keyring_%d", k)
	}
}

func (k Kind) HumanName() string {
	switch k {
	case KindYubikey:
		return "Yubikey"
	case KindKeychain:
		return "Keychain"
	case KindFiles:
		return "Files"
	case KindNone:
		return "none"
	default:
		return fmt.Sprintf("keyring_%d", k)
	}
}

func (k *Kind) fromString(v string) error {
	switch v {
	case "yubikey":
		*k = KindYubikey
	case "keychain":
		*k = KindKeychain
	case "files":
		*k = KindFiles
	case "none":
		*k = KindNone
	default:
		return fmt.Errorf("unknown keyring: %s", v)
	}

	return nil
}

func (k Kind) MarshalYAML() (interface{}, error) {
	return k.String(), nil
}

func (k *Kind) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return k.fromString(s)
}

func (k Kind) MarshalJSON() ([]byte, error) {
	return json.Marshal(k.String())
}

func (k *Kind) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return k.fromString(s)
}
