package memring

import (
	"crypto"
	"crypto/x509"
	"fmt"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/skotty/internal/certgen"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
	"a.yandex-team.ru/security/skotty/skotty/pkg/softattest"
)

const (
	TokenType = skotty.TokenTypeMemory
	Name      = "memory"
	HumanName = "Memory"
)

var supportedKeyTypes = []keyring.KeyPurpose{
	keyring.KeyPurposeSudo,
	keyring.KeyPurposeSecure,
	keyring.KeyPurposeInsecure,
}

var _ keyring.Keyring = (*MemRing)(nil)
var _ keyring.Tx = (*Tx)(nil)

type MemRing struct {
	store *MemStore
}

type Tx struct {
	store *MemStore
}

func NewMemring() *MemRing {
	return &MemRing{
		store: NewMemStore(),
	}
}

func (k *MemRing) TokenType() skotty.TokenType {
	return TokenType
}

func (k *MemRing) Name() string {
	return Name
}

func (k *MemRing) HumanName() string {
	return HumanName
}

func (k *MemRing) SupportedKeyTypes() []keyring.KeyPurpose {
	return supportedKeyTypes
}

func (k *MemRing) IsTouchableKey(_ keyring.KeyPurpose) bool {
	return false
}

func (k *MemRing) Serial() (string, error) {
	return softattest.SharedAttestator().TokenSerial()
}

func (k *MemRing) PinStoreOpts() []pinstore.Option {
	return nil
}

func (k *MemRing) Tx() (keyring.Tx, error) {
	return &Tx{
		store: k.store,
	}, nil
}

func (k *MemRing) Close() {}

func (t *Tx) Serial() (string, error) {
	return softattest.SharedAttestator().TokenSerial()
}

func (t *Tx) TokenType() skotty.TokenType {
	return TokenType
}

func (t *Tx) Name() string {
	return Name
}

func (t *Tx) HumanName() string {
	return HumanName
}

func (t *Tx) SupportedKeyTypes() []keyring.KeyPurpose {
	return supportedKeyTypes
}

func (t *Tx) AttestationCertificate() (*x509.Certificate, error) {
	return softattest.SharedAttestator().Certificate()
}

func (t *Tx) Attest(keyType keyring.KeyPurpose) (*x509.Certificate, error) {
	keyPair, err := t.store.GetCert(keyType)
	if err != nil {
		return nil, fmt.Errorf("failed to fetch keypair: %w", err)
	}

	pubKey, err := x509.ParseCertificate(keyPair.PubKey)
	if err != nil {
		return nil, fmt.Errorf("failed to parse certificate: %w", err)
	}

	return softattest.SharedAttestator().Attest(pubKey, softattest.PINPolicyOnce, softattest.TouchPolicyNever)
}

func (t *Tx) RenewCertificate(keyType keyring.KeyPurpose) (*x509.Certificate, error) {
	return t.GenCertificate(keyType)
}

func (t *Tx) GenCertificate(keyType keyring.KeyPurpose) (*x509.Certificate, error) {
	var keyPair KeyPair
	var err error
	keyPair.PubKey, keyPair.PrivKey, err = certgen.GenCertificate(keyType)
	if err != nil {
		return nil, fmt.Errorf("failed to generate new keypair: %w", err)
	}

	err = t.store.SaveCert(keyType, keyPair)
	if err != nil {
		return nil, fmt.Errorf("failed to save keypair: %w", err)
	}

	return x509.ParseCertificate(keyPair.PubKey)
}

func (t *Tx) SetCertificate(keyType keyring.KeyPurpose, crt *x509.Certificate) error {
	keyPair, err := t.store.GetCert(keyType)
	if err != nil {
		return fmt.Errorf("failed to fetch keypair: %w", err)
	}

	keyPair.PubKey = crt.Raw
	return t.store.SaveCert(keyType, keyPair)
}

func (t *Tx) Certificate(keyType keyring.KeyPurpose) (*x509.Certificate, error) {
	keyPair, err := t.store.GetCert(keyType)
	if err != nil {
		return nil, fmt.Errorf("failed to fetch keypair: %w", err)
	}

	return x509.ParseCertificate(keyPair.PubKey)
}

func (t *Tx) Signer(keyType keyring.KeyPurpose) (crypto.Signer, error) {
	keyPair, err := t.store.GetCert(keyType)
	if err != nil {
		return nil, fmt.Errorf("failed to fetch keypair: %w", err)
	}

	switch keyType {
	case keyring.KeyPurposeLegacy:
		return x509.ParsePKCS1PrivateKey(keyPair.PrivKey)
	default:
		return x509.ParseECPrivateKey(keyPair.PrivKey)
	}
}

func (t *Tx) Close() {}
