package keyring

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*TouchPolicy)(nil)
var _ yaml.Marshaler = (*TouchPolicy)(nil)
var _ json.Unmarshaler = (*TouchPolicy)(nil)
var _ json.Marshaler = (*TouchPolicy)(nil)

type TouchPolicy uint8

const (
	TouchPolicyNone TouchPolicy = iota
	TouchPolicyNever
	TouchPolicyCached
	TouchPolicyAlways
)

func (p TouchPolicy) String() string {
	switch p {
	case TouchPolicyNone:
		return "none"
	case TouchPolicyNever:
		return "never"
	case TouchPolicyAlways:
		return "always"
	case TouchPolicyCached:
		return "cached"
	default:
		return fmt.Sprintf("touch_policy_%d", p)
	}
}

func (p *TouchPolicy) fromString(v string) error {
	switch v {
	case "none":
		*p = TouchPolicyNone
	case "never":
		*p = TouchPolicyNever
	case "always":
		*p = TouchPolicyAlways
	case "cached":
		*p = TouchPolicyCached
	default:
		return fmt.Errorf("unknown touch policy: %s", v)
	}

	return nil
}

func (p TouchPolicy) MarshalYAML() (interface{}, error) {
	return p.String(), nil
}

func (p *TouchPolicy) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return p.fromString(s)
}

func (p TouchPolicy) MarshalJSON() ([]byte, error) {
	return json.Marshal(p.String())
}

func (p *TouchPolicy) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return p.fromString(s)
}
