package anypin

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/keychain"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/memcache"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/pinentry"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/plain"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/secret"
)

var _ yaml.Unmarshaler = (*Provider)(nil)
var _ yaml.Marshaler = (*Provider)(nil)
var _ json.Unmarshaler = (*Provider)(nil)
var _ json.Marshaler = (*Provider)(nil)
var _ pinstore.Provider = (*Provider)(nil)

type Provider struct {
	pinstore.Provider
}

func NewProvider(source string) (*Provider, error) {
	var p Provider
	if err := p.fromSource(source); err != nil {
		return nil, err
	}

	return &p, nil
}

func (p *Provider) fromSource(source string) error {
	k := pinstore.KindNone
	if err := k.FromSource(source); err != nil {
		return fmt.Errorf("unable to parse pinentry provider: %w", err)
	}

	var provider pinstore.Provider
	var err error
	switch k {
	case pinstore.KindPlain:
		provider, err = plain.NewProvider(source)
	case pinstore.KindSecret:
		provider, err = secret.NewProvider(source)
	case pinstore.KindPinentry:
		provider, err = pinentry.NewProvider(source)
		if err == nil {
			provider, err = memcache.NewProvider(provider)
		}
	case pinstore.KindKeychain:
		provider, err = keychain.NewProvider()
		if err == nil {
			provider, err = memcache.NewProvider(provider)
		}
	default:
		err = fmt.Errorf("unsupported pin provider: %s", k)
	}

	if err != nil {
		return err
	}

	p.Provider = provider
	return nil
}

func (p *Provider) Source() (string, error) {
	source, err := p.Provider.Source()
	if err != nil {
		return "", err
	}

	return fmt.Sprintf("%s:%s", p.Kind(), source), nil
}

func (p Provider) MarshalYAML() (interface{}, error) {
	return p.Source()
}

func (p *Provider) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return p.fromSource(s)
}

func (p Provider) MarshalJSON() ([]byte, error) {
	source, err := p.Source()
	if err != nil {
		return nil, err
	}

	return json.Marshal(source)
}

func (p *Provider) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return p.fromSource(s)
}
