package keychain

import (
	"errors"
	"fmt"

	"a.yandex-team.ru/security/skotty/skotty/internal/osxkeychain"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
)

var _ pinstore.Provider = (*Provider)(nil)

const keychainService = "skotty"

type Provider struct{}

func NewProvider() (*Provider, error) {
	return &Provider{}, nil
}

func (p *Provider) CanStore() bool {
	return true
}

func (p *Provider) Source() (string, error) {
	return keychainService, nil
}

func (p *Provider) Kind() pinstore.Kind {
	return pinstore.KindKeychain
}

func (p *Provider) StorePIN(pin string, opts ...pinstore.Option) error {
	return p.StorePassphrase(pin, opts...)
}

func (p *Provider) GetPIN(validator pinstore.PassphraseValidator, opts ...pinstore.Option) (string, error) {
	return p.GetPassphrase(validator, opts...)
}

func (p *Provider) StorePassphrase(passphrase string, opts ...pinstore.Option) error {
	item := osxkeychain.NewItem()
	item.SetSecClass(osxkeychain.SecClassGenericPassword)
	item.SetService(keychainService)
	item.SetAccessible(osxkeychain.AccessibleWhenUnlocked)
	item.SetData([]byte(passphrase))
	for _, opt := range opts {
		switch o := opt.(type) {
		case pinstore.OptionSerial:
			keyAccountName := accountName(o.Serial)
			item.SetAccount(keyAccountName)
			item.SetLabel(keyAccountName)
		case pinstore.OptionSync:
			if o.Enabled {
				item.SetSynchronizable(osxkeychain.SynchronizableYes)
			} else {
				item.SetSynchronizable(osxkeychain.SynchronizableNo)
			}
		}
	}

	err := osxkeychain.AddItem(item)
	if err != nil {
		if errors.Is(err, osxkeychain.ErrorDuplicateItem) {
			item.SetLabel("")
			item.SetSynchronizable(osxkeychain.SynchronizableDefault)
			item.SetMatchLimit(osxkeychain.MatchLimitOne)
			item.SetData(nil)

			update := osxkeychain.NewItem()
			update.SetData([]byte(passphrase))
			update.SetAccessible(osxkeychain.AccessibleWhenUnlocked)
			return osxkeychain.UpdateItem(item, update)
		}
		return err
	}

	return nil
}

func (p *Provider) GetPassphrase(validator pinstore.PassphraseValidator, opts ...pinstore.Option) (string, error) {
	var serial string
	for _, opt := range opts {
		switch o := opt.(type) {
		case pinstore.OptionSerial:
			serial = o.Serial
		}
	}

	name := accountName(serial)
	query := osxkeychain.NewItem()
	query.SetSecClass(osxkeychain.SecClassGenericPassword)
	query.SetService(keychainService)
	query.SetMatchLimit(osxkeychain.MatchLimitOne)
	query.SetReturnData(true)
	query.SetAccount(name)
	results, err := osxkeychain.QueryItem(query)
	if err != nil {
		return "", err
	}

	if len(results) > 1 {
		return "", fmt.Errorf("too many results for keychain entry %q: %d", name, len(results))
	}

	if len(results) == 0 {
		return "", fmt.Errorf("item not found in keychain: %s", name)
	}

	passphrase := string(results[0].Data)
	if err := validator(passphrase); err != nil {
		return "", pinstore.UnwrapPermanent(err)
	}

	return passphrase, nil
}

func accountName(serial string) string {
	return fmt.Sprintf("%s@skotty-pinstore", serial)
}
