package pinentry

import (
	"errors"
	"fmt"
	"strings"
	"sync"

	"a.yandex-team.ru/security/skotty/libs/pinentry"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
)

var _ pinstore.Provider = (*Provider)(nil)

type Provider struct {
	program string
	args    []string
	mu      sync.Mutex
}

func NewProvider(source string) (*Provider, error) {
	//TODO(buglloc): allow to pass additional arguments ("C:\\Program Files\Somethig" make bo-bo)
	return &Provider{
		program: pinstore.KindPinentry.TrimSource(source),
	}, nil
}

func (p *Provider) CanStore() bool {
	return false
}

func (p *Provider) Source() (string, error) {
	cmdline := p.program
	if len(p.args) > 0 {
		cmdline += " " + strings.Join(p.args, " ")
	}

	return cmdline, nil
}

func (p *Provider) Kind() pinstore.Kind {
	return pinstore.KindPinentry
}

func (p *Provider) StorePIN(_ string, _ ...pinstore.Option) error {
	return errors.New("not supported")
}

func (p *Provider) GetPIN(validator pinstore.PassphraseValidator, opts ...pinstore.Option) (string, error) {
	pinOpts, serial, err := parseOpts(opts...)
	if err != nil {
		return "", err
	}

	pinOpts = append(
		[]pinentry.ClientOption{
			pinentry.WithTitle("Skotty pinentry"),
			pinentry.WithPrompt("Enter PIN"),
		},
		pinOpts...,
	)

	return p.callPinEntry(validator, serial, pinOpts...)
}

func (p *Provider) StorePassphrase(_ string, _ ...pinstore.Option) error {
	return errors.New("not supported")
}

func (p *Provider) GetPassphrase(validator pinstore.PassphraseValidator, opts ...pinstore.Option) (string, error) {
	pinOpts, serial, err := parseOpts(opts...)
	if err != nil {
		return "", err
	}

	pinOpts = append(
		[]pinentry.ClientOption{
			pinentry.WithTitle("Skotty pinentry"),
			pinentry.WithPrompt("Enter Passphrase"),
		},
		pinOpts...,
	)

	return p.callPinEntry(validator, serial, pinOpts...)
}

func (p *Provider) callPinEntry(validator pinstore.PassphraseValidator, _ string, opts ...pinentry.ClientOption) (string, error) {
	p.mu.Lock()
	defer p.mu.Unlock()

	check := func(pass string) (bool, error) {
		err := validator(pass)
		if err == nil {
			return true, nil
		}

		if pinstore.IsPermanent(err) {
			return true, pinstore.UnwrapPermanent(err)
		}

		return false, err
	}

	client, err := pinentry.NewClient(
		append(
			[]pinentry.ClientOption{
				pinentry.WithBinaryName(p.program),
				pinentry.WithArgs(p.args),
				pinentry.WithGPGTTY(),
			},
			opts...,
		)...,
	)
	if err != nil {
		return "", fmt.Errorf("unable to call pinentry: %w", err)
	}

	defer func() { _ = client.Close() }()

	for {
		pin, _, err := client.GetPIN()
		if err != nil {
			return "", err
		}

		stop, err := check(pin)
		if stop {
			return pin, err
		}

		_ = client.Error(err.Error())
	}
}

func parseOpts(opts ...pinstore.Option) ([]pinentry.ClientOption, string, error) {
	out := make([]pinentry.ClientOption, len(opts))
	var serial string
	for i, opt := range opts {
		switch o := opt.(type) {
		case pinstore.OptionSerial:
			out[i] = pinentry.WithKeyInfo("skotty keyring #" + o.Serial)
			serial = o.Serial
		case pinstore.OptionDescription:
			out[i] = pinentry.WithDesc(o.Description)
		case pinstore.OptionSync:
			// nothing to do
		default:
			return nil, "", fmt.Errorf("unsupported option: %T", opt)
		}
	}

	return out, serial, nil
}
