package pinstore

import (
	"fmt"
	"strings"
)

type Kind uint8

const (
	KindNone Kind = iota
	KindPlain
	KindSecret
	KindPinentry
	KindKeychain
)

func (k Kind) String() string {
	switch k {
	case KindNone:
		return ""
	case KindPlain:
		return "plain"
	case KindSecret:
		return "secret"
	case KindPinentry:
		return "pinentry"
	case KindKeychain:
		return "keychain"
	default:
		return fmt.Sprintf("unknown_%d", k)
	}
}

func (k Kind) Description() string {
	switch k {
	case KindNone:
		return ""
	case KindPlain:
		return "Plain - will save unencrypted"
	case KindSecret:
		return "Secret - will save encrypted on the machine id key"
	case KindPinentry:
		return "Pinentry - will ask througth pinentry program"
	case KindKeychain:
		return "Keychain - will save into keychain"
	default:
		return fmt.Sprintf("unknown_%d", k)
	}
}

func (k *Kind) FromString(s string) error {
	switch s {
	case "":
		*k = KindNone
	case "plain":
		*k = KindPlain
	case "secret":
		*k = KindSecret
	case "pinentry":
		*k = KindPinentry
	case "keychain":
		*k = KindKeychain
	default:
		return fmt.Errorf("unknown provider: %s", s)
	}
	return nil
}

func (k *Kind) FromSource(source string) error {
	idx := strings.IndexByte(source, ':')

	*k = KindNone
	if idx > 0 {
		return k.FromString(source[:idx])
	}
	return nil
}

func (k Kind) TrimSource(source string) string {
	return strings.TrimPrefix(source, k.String()+":")
}
