package procutil

import (
	"bytes"
	"errors"
	"fmt"
	"os/exec"
	"strings"
	"syscall"
)

type CmdResult struct {
	ExitCode int
	Stdout   string
	Stderr   string
}

func RunCommand(name string, args ...string) (*CmdResult, error) {
	cmd := exec.Command(name, args...)
	return RunCmd(cmd)
}

func RunCmd(cmd *exec.Cmd) (*CmdResult, error) {
	if cmd.Stdout != nil {
		return nil, errors.New("stdout already set")
	}
	if cmd.Stderr != nil {
		return nil, errors.New("stderr already set")
	}

	var stdout, stderr bytes.Buffer
	cmd.Stdout = &stdout
	cmd.Stderr = &stderr
	var exitCode int
	if err := cmd.Run(); err != nil {
		if exitErr, ok := err.(*exec.ExitError); ok {
			// The program has exited with an exit code != 0
			if status, ok := exitErr.Sys().(syscall.WaitStatus); ok {
				exitCode = status.ExitStatus()
			} else {
				return nil, fmt.Errorf("%s: wait error: %w", cmd, err)
			}
		} else {
			return nil, fmt.Errorf("%s: wait error: %w", cmd, err)
		}
	}

	return &CmdResult{
		ExitCode: exitCode,
		Stdout:   strings.TrimSpace(stdout.String()),
		Stderr:   strings.TrimSpace(stderr.String()),
	}, nil
}
