package asker

import (
	"errors"
	"fmt"
	"os/exec"
	"reflect"

	"github.com/AlecAivazis/survey/v2"
	"golang.org/x/sys/execabs"

	"a.yandex-team.ru/security/skotty/libs/pinentry"
	"a.yandex-team.ru/security/skotty/skotty/internal/executil"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/passutil"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/yubikey"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil/sshclient"
)

type YubiProvisionAction uint8

const (
	YubiProvisionActionNone YubiProvisionAction = iota
	YubiProvisionActionReset
	YubiProvisionActionPIN
)

func SelectKeyring(kinds ...keyring.Kind) (keyring.Kind, error) {
	switch len(kinds) {
	case 0:
		return keyring.KindNone, errors.New("no keyring available")
	case 1:
		return kinds[0], nil
	}

	opts := make([]string, len(kinds))
	for i, c := range kinds {
		opts[i] = c.HumanName()
	}

	prompt := &survey.Select{
		Message: "Select Keyring:",
		Options: opts,
	}

	var idx int
	if err := survey.AskOne(prompt, &idx); err != nil {
		return keyring.KindNone, err
	}

	return kinds[idx], nil
}

func SelectYubikey(cards ...yubikey.Card) (yubikey.Card, error) {
	switch len(cards) {
	case 0:
		return yubikey.Card{}, errors.New("no cards")
	case 1:
		return cards[0], nil
	}

	opts := make([]string, len(cards))
	for i, c := range cards {
		opts[i] = c.String()
	}

	prompt := &survey.Select{
		Message: "Select Yubikey:",
		Options: opts,
	}

	var idx int
	if err := survey.AskOne(prompt, &idx); err != nil {
		return yubikey.Card{}, err
	}

	return cards[idx], nil
}

func SelectKeychainCollection(collections ...string) (string, bool, error) {
	askNewCollection := func() (string, bool, error) {
		prompt := &survey.Input{
			Message: "Enter new collection name",
			Default: "<use default>",
		}

		var out string
		if err := survey.AskOne(prompt, &out); err != nil {
			return "", false, err
		}

		if out == "<use default>" {
			return "", false, nil
		}

		return out, true, nil
	}

	if len(collections) == 0 {
		return askNewCollection()
	}

	opts := collections[:]
	isNew := len(opts)
	opts = append(opts, "Create new one")

	prompt := &survey.Select{
		Message: "Select collection:",
		Options: opts,
	}

	var idx int
	if err := survey.AskOne(prompt, &idx); err != nil {
		return "", false, err
	}

	if idx == isNew {
		return askNewCollection()
	}
	return opts[idx], false, nil
}

func SelectSSHClients(clients ...sshclient.ClientKind) ([]sshclient.ClientKind, error) {
	opts := make([]string, len(clients))
	defaults := make([]int, len(clients))
	for i, c := range clients {
		opts[i] = c.String()
		defaults[i] = i
	}

	prompt := &survey.MultiSelect{
		Message: "Select the SSH clients that you are use:",
		Options: opts,
		Default: defaults,
	}

	var items []int
	if err := survey.AskOne(prompt, &items); err != nil {
		return nil, err
	}

	out := make([]sshclient.ClientKind, len(items))
	for i, idx := range items {
		out[i] = clients[idx]
	}

	return out, nil
}

func SelectPinstore(_ string, kinds ...pinstore.Kind) (pinstore.Kind, error) {
	//TODO(buglloc) enable me back (or not, think about it)
	return kinds[0], nil
	//options := make([]string, len(kinds))
	//for i, k := range kinds {
	//	options[i] = k.Description()
	//}
	//
	//prompt := &survey.Select{
	//	Message: fmt.Sprintf("Choose how to store %s:", target),
	//	Options: options,
	//}
	//
	//var idx int
	//if err := survey.AskOne(prompt, &idx); err != nil {
	//	return pinstore.KindNone, err
	//}
	//
	//return kinds[idx], nil
}

func SelectPinentry() (string, error) {
	var defBin string
	for _, bin := range pinentry.DefaultProgs {
		if p, err := execabs.LookPath(bin); err == nil {
			defBin = p
			break
		}
	}

	prompt := &survey.Input{
		Message: "Please provide pinentry program to use",
		Default: defBin,
		Suggest: func(toComplete string) []string {
			variants, _ := executil.SuggestExecutables(toComplete)
			return variants
		},
	}

	validator := func(ans interface{}) error {
		file, ok := ans.(string)
		if !ok {
			return fmt.Errorf("cannot pinetnry program of type %v", reflect.TypeOf(ans).Name())
		}

		_, err := exec.LookPath(file)
		return err
	}

	var bin string
	err := survey.AskOne(prompt, &bin,
		survey.WithShowCursor(true),
		survey.WithValidator(validator),
	)
	if err != nil {
		return "", err
	}

	bin, err = exec.LookPath(bin)
	return bin, err
}

func ChooseYubiProvisionAction() (YubiProvisionAction, error) {
	actions := []string{
		"Enter PIN",
		"Reset",
	}

	prompt := &survey.Select{
		Message: "Seems your yubikey already provisioned, choose next step:",
		Options: actions,
	}

	var idx int
	if err := survey.AskOne(prompt, &idx); err != nil {
		return YubiProvisionActionNone, err
	}

	if idx == 0 {
		return YubiProvisionActionPIN, nil
	}

	return YubiProvisionActionReset, nil
}

func ConfirmOpenInBrowser() (bool, error) {
	prompt := &survey.Confirm{
		Message: "Open it in your default browser?",
		Default: true,
	}

	var ok bool
	err := survey.AskOne(prompt, &ok)
	return ok, err
}

func ConfirmReset(card yubikey.Card) (bool, error) {
	prompt := &survey.Confirm{
		Message: fmt.Sprintf("Confirm reseting %q?", card),
		Default: true,
	}

	var ok bool
	err := survey.AskOne(prompt, &ok)
	return ok, err
}

func ConfirmInstallService() (bool, error) {
	prompt := &survey.Confirm{
		Message: "Enable skotty service?",
		Default: true,
	}

	var ok bool
	err := survey.AskOne(prompt, &ok)
	return ok, err
}

func ConfirmSocketNotifications() (bool, error) {
	prompt := &survey.Confirm{
		Message: "Enable usage notifications?",
		Default: false,
	}

	var ok bool
	err := survey.AskOne(prompt, &ok)
	return ok, err
}

func AskYubiPIN(message string) (string, error) {
	if message != "" {
		message += ". "
	}

	return askPassphrase(&survey.Password{
		Message: message + "Enter yubikey PIN:",
	})
}

func AskPUK(message string) (string, error) {
	if message != "" {
		message += ". "
	}

	return askPassphrase(&survey.Password{
		Message: message + "Enter yubikey PUK:",
	})
}

func AskNewPIN() (string, error) {
	for {
		passphrase, err := askPassphrase(
			&survey.Password{
				Message: "Enter new PIN (6-8 characters):",
			},
			survey.MinLength(6),
			survey.MaxLength(8),
			pinValidator("PIN"),
		)
		if err != nil {
			return "", err
		}

		_, err = askPassphrase(
			&survey.Password{
				Message: "Enter same PIN again (Ctrl+C to try from the beginning):",
			},
			sameValValidator("PIN", passphrase),
		)
		if err != nil {
			continue
		}

		return passphrase, nil
	}
}

func AskNewPUK() (string, error) {
	for {
		passphrase, err := askPassphrase(
			&survey.Password{
				Message: "Enter new PUK (6-8 characters):",
			},
			survey.MinLength(6),
			survey.MaxLength(8),
			pinValidator("PUK"),
		)
		if err != nil {
			return "", err
		}

		_, err = askPassphrase(
			&survey.Password{
				Message: "Enter same PUK again (Ctrl+C to try from the beginning):",
			},
			sameValValidator("PUK", passphrase),
		)
		if err != nil {
			continue
		}

		return passphrase, nil
	}
}

func ConfirmProvidePIN() (bool, error) {
	prompt := &survey.Confirm{
		Message: "Do you want to set PIN/PUK manually? (otherwise it will be generated automatically)",
		Default: false,
	}

	var ok bool
	err := survey.AskOne(prompt, &ok)
	return ok, err
}

func ConfirmProvidePUK() (bool, error) {
	prompt := &survey.Confirm{
		Message: "Do you want to set PUK manually? (otherwise it will be generated automatically)",
		Default: false,
	}

	var ok bool
	err := survey.AskOne(prompt, &ok)
	return ok, err
}

func ConfirmProvidePassphrase() (bool, error) {
	prompt := &survey.Confirm{
		Message: "Do you want to set passphrase manually? (otherwise it will be generated automatically)",
		Default: false,
	}

	var ok bool
	err := survey.AskOne(prompt, &ok)
	return ok, err
}

func AskNewPassphrase() (string, error) {
	for {
		passphrase, err := askPassphrase(
			&survey.Password{
				Message: "Enter new passphrase (at least 8 characters):",
			},
			survey.MinLength(8),
			passphraseValidator,
		)
		if err != nil {
			return "", err
		}

		_, err = askPassphrase(
			&survey.Password{
				Message: "Enter same passphrase again (Ctrl+C to try from the beginning):",
			},
			sameValValidator("passphrase", passphrase),
		)
		if err != nil {
			continue
		}

		return passphrase, nil
	}
}

func AskFileringPath(baseBase string) (string, error) {
	prompt := &survey.Input{
		Message: "Path to store keys:",
		Default: baseBase,
	}

	err := survey.AskOne(prompt, &baseBase, survey.WithValidator(survey.Required))
	return baseBase, err
}

func askPassphrase(prompt *survey.Password, validators ...survey.Validator) (string, error) {
	pinValidators := []survey.Validator{
		survey.Required,
	}
	pinValidators = append(pinValidators, validators...)

	var pin string
	err := survey.AskOne(prompt, &pin, survey.WithValidator(survey.ComposeValidators(pinValidators...)))
	return pin, err
}

func sameValValidator(kind, val string) survey.Validator {
	return func(cur interface{}) error {
		str, ok := cur.(string)
		if !ok {
			return fmt.Errorf("cannot check same val on response of type %v", reflect.TypeOf(cur).Name())
		}

		if str != val {
			return fmt.Errorf("%s mismatch", kind)
		}

		return nil
	}
}

func pinValidator(kind string) survey.Validator {
	return func(cur interface{}) error {
		pin, ok := cur.(string)
		if !ok {
			return fmt.Errorf("cannot check pin/puk on response of type %v", reflect.TypeOf(cur).Name())
		}

		// default PIN
		if pin == "123456" {
			return fmt.Errorf("%s can't equal to default Yubikey PIN", kind)
		}

		// default PUK
		if pin == "12345678" {
			return fmt.Errorf("%s can't equal to default Yubikey PUK", kind)
		}

		return passutil.CheckPassword(pin,
			passutil.IsConsecutivelyRepeated(kind),
			passutil.IsSame(kind),
		)
	}
}

func passphraseValidator(cur interface{}) error {
	pass, ok := cur.(string)
	if !ok {
		return fmt.Errorf("cannot check passphrase on response of type %v", reflect.TypeOf(cur).Name())
	}

	return passutil.CheckPassword(pass,
		passutil.IsConsecutivelyRepeated("passphrase"),
		passutil.IsSame("passphrase"),
	)
}
