package scenario

import (
	"fmt"

	"a.yandex-team.ru/security/skotty/libs/certutil"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/setuputil"
)

func keyringCall(kr keyring.Keyring, fn func(tx keyring.Tx) error) error {
	tx, err := kr.Tx()
	if err != nil {
		return fmt.Errorf("can't open keyring session: %w", err)
	}
	defer tx.Close()

	return fn(tx)
}

func csrFromTokenCerts(tx keyring.Tx, certs []TokenCert) ([]skotty.RequestedCertificate, error) {
	fillCertInfo := func(out *skotty.RequestedCertificate, cert TokenCert) error {
		attestCert, err := tx.Attest(cert.KeyType)
		if err != nil {
			return fmt.Errorf("failed to attest keytype %q: %w", cert.KeyType, err)
		}

		out.AttestationCert = certutil.CertToPem(attestCert)
		out.Cert = certutil.CertToPem(cert.Cert)
		return nil
	}

	out := make([]skotty.RequestedCertificate, 0, len(certs))
	for i, c := range certs {
		csr := skotty.RequestedCertificate{
			HostID: i,
			Type:   setuputil.KeyPurposeToCertType(c.KeyType),
		}

		if err := fillCertInfo(&csr, c); err != nil {
			return nil, err
		}

		out = append(out, csr)
	}

	return out, nil
}

func parseIssuedCerts(tokenCerts []TokenCert, issuedCerts []skotty.IssuedCertificate) ([]SignedCert, error) {
	out := make([]SignedCert, 0, len(tokenCerts))
	for hostID, csr := range tokenCerts {
		found := false
		for _, cert := range issuedCerts {
			if cert.HostID != hostID {
				continue
			}

			found = true
			crt, err := certutil.PemToCert(cert.Cert)
			if err != nil {
				return out, fmt.Errorf("failed to parse issued certificate for key %q: %w", csr.KeyType, err)
			}

			out = append(out, SignedCert{
				KeyType: csr.KeyType,
				Cert:    crt,
				SSHCert: cert.SSHCert,
			})
		}

		if !found {
			return out, fmt.Errorf("enrollment doesn't returns cert for key: %s", csr.KeyType)
		}
	}

	return out, nil
}
