package scenario

import (
	"fmt"
	"io"
	"os"
	"runtime"
	"sync"

	"golang.org/x/crypto/ssh/terminal"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/pubstore"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/asker"
	"a.yandex-team.ru/security/skotty/skotty/internal/version"
	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
)

type logKind rune

const (
	logKindInfo    logKind = 'i'
	logKindWait    logKind = '.'
	logKindSuccess logKind = '+'
	logKindError   logKind = '-'
	logKindWarn    logKind = '!'
)

type Scenario struct {
	logOut      io.Writer
	colored     bool
	pubStore    pubstore.PubStore
	svcUpstream string
	svc         *skotty.EnrollClient
	svcOnce     sync.Once
}

func NewScenario(opts ...Option) *Scenario {
	out := Scenario{
		logOut:      os.Stdout,
		svcUpstream: skotty.DefaultUpstream,
	}

	for _, opt := range opts {
		switch v := opt.(type) {
		case logOutOption:
			out.logOut = v.out
		case enrollUpstreamOption:
			out.svcUpstream = v.upstream
		case pubStoreOption:
			out.pubStore = v.pubStore
		}
	}

	if f, ok := out.logOut.(interface{ Fd() uintptr }); ok {
		out.colored = runtime.GOOS != "windows" && terminal.IsTerminal(int(f.Fd()))
	}

	return &out
}

func (s *Scenario) skottyService() *skotty.EnrollClient {
	s.svcOnce.Do(func() {
		svc := skotty.NewEnrollClient(
			skotty.WithUpstream(s.svcUpstream),
			skotty.WithClientVersion(version.Full()),
		)

		s.svc = svc
	})

	return s.svc
}

func (s *Scenario) LogInfo(format string, args ...interface{}) {
	s.log(logKindInfo, format, args...)
}

func (s *Scenario) LogWait(format string, args ...interface{}) {
	s.log(logKindWait, format, args...)
}

func (s *Scenario) LogSuccess(format string, args ...interface{}) {
	s.log(logKindSuccess, format, args...)
}

func (s *Scenario) LogError(format string, args ...interface{}) {
	s.log(logKindError, format, args...)
}

func (s *Scenario) LogWarn(format string, args ...interface{}) {
	s.log(logKindWarn, format, args...)
}

func (s *Scenario) OpenInBrowser(url string, next func() error) error {
	ok, err := asker.ConfirmOpenInBrowser()
	if err != nil {
		return err
	}

	if !ok {
		return next()
	}

	if err := osutil.OpenInBrowser(url); err != nil {
		s.LogError("fail: %v", err)
		s.LogInfo("you must to follow URL manually")
	}
	return next()
}

func (s *Scenario) log(sign logKind, format string, args ...interface{}) {
	if !s.colored {
		_, _ = fmt.Fprintf(s.logOut, fmt.Sprintf("[%c] %s\n", sign, format), args...)
		return
	}

	color := 15
	switch sign {
	case logKindError:
		color = 9
	case logKindWarn:
		color = 11
	case logKindSuccess:
		color = 10
	}

	_, _ = fmt.Fprintf(s.logOut, fmt.Sprintf("\x1B[38;5;%dm[%c]\u001B[0m %s\n", color, sign, format), args...)
}
