package scenario

import (
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/filering"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/keychain"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring/yubiring"
	"a.yandex-team.ru/security/skotty/skotty/internal/passutil"
	"a.yandex-team.ru/security/skotty/skotty/internal/paths"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore"
	pinchain "a.yandex-team.ru/security/skotty/skotty/internal/pinstore/keychain"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/pinentry"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/plain"
	"a.yandex-team.ru/security/skotty/skotty/internal/pinstore/secret"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/asker"
	"a.yandex-team.ru/security/skotty/skotty/internal/yubikey"
	"a.yandex-team.ru/security/skotty/skotty/pkg/osutil"
)

type Setup struct {
	*Enroll
}

type YubiringConf struct {
	Serial  uint32
	PIN     string
	userPIN bool
	PUK     string
	userPUK bool
}

func (y *YubiringConf) MaskedPIN() string {
	if y.PIN == "" {
		return ""
	}

	if y.userPIN {
		return passutil.Mask(y.PIN)
	}

	return y.PIN
}

func (y *YubiringConf) MaskedPUK() string {
	if y.PUK == "" {
		return ""
	}

	if y.userPUK {
		return passutil.Mask(y.PUK)
	}

	return y.PUK
}

type KeychainConf struct {
	Collection string
}

type FileringConf struct {
	BasePath   string
	Passphrase string
}

func NewSetup(opts ...Option) *Setup {
	return &Setup{
		Enroll: NewEnroll(opts...),
	}
}

func (s *Setup) SelectKeyring(yubikeyOnly bool, next func(keyring.Kind) error) error {
	s.LogInfo("initialize keyring")

	keyrings := supportedKeyrings
	if yubikeyOnly {
		keyrings = []keyring.Kind{keyring.KindYubikey}
	}

	var availableKeyrings []keyring.Kind
	for _, typ := range keyrings {
		var isAvailable bool
		var err error
		switch typ {
		case keyring.KindYubikey:
			isAvailable, err = yubiring.IsAvailable()
		case keyring.KindKeychain:
			isAvailable, err = keychain.IsAvailable()
		case keyring.KindFiles:
			isAvailable = true
			err = nil
		default:
			return fmt.Errorf("unsupported keyring: %s", typ)
		}

		if err != nil {
			s.LogWarn("unable to use %q keyring: %v", typ, err)
			continue
		}

		if !isAvailable {
			continue
		}

		availableKeyrings = append(availableKeyrings, typ)
	}

	typ, err := asker.SelectKeyring(availableKeyrings...)
	if err != nil {
		return err
	}

	return next(typ)
}

func (s *Setup) SelectPinstore(kind keyring.Kind, next func(pinstore.Provider) error) error {
	s.LogInfo("initialize pinstore")

	var targetKind pinstore.Kind
	var err error
	switch kind {
	case keyring.KindYubikey:
		targetKind, err = asker.SelectPinstore("PIN", availablePinstore...)
	case keyring.KindFiles:
		targetKind, err = asker.SelectPinstore("Passphrase", availablePinstore...)
	case keyring.KindKeychain:
		return next(nil)
	default:
		return fmt.Errorf("unsupported keyring: %s", kind)
	}

	if err != nil {
		return err
	}

	return s.SelectPinSource(targetKind, next)
}

func (s *Setup) SelectPinSource(kind pinstore.Kind, next func(pinstore.Provider) error) error {
	var provider pinstore.Provider
	var err error
	switch kind {
	case pinstore.KindSecret:
		provider, err = secret.NewProvider("")
	case pinstore.KindKeychain:
		provider, err = pinchain.NewProvider()
	case pinstore.KindPinentry:
		s.LogInfo("initialize pinentry")
		var source string
		source, err = asker.SelectPinentry()
		if err == nil {
			provider, err = pinentry.NewProvider(source)
		}
	default:
		return fmt.Errorf("unsupported pinstore: %s", kind)
	}

	if err != nil {
		return err
	}

	return next(provider)
}

func (s *Setup) NewYubikey(next func(YubiringConf, keyring.Keyring) error) error {
	s.LogInfo("initialize %s", yubiring.HumanName)

	cards, err := yubikey.Cards()
	if err != nil {
		return fmt.Errorf("failed to list yubikeys: %w", err)
	}

	card, err := asker.SelectYubikey(cards...)
	if err != nil {
		return err
	}

	err = s.tryOpenYubikey(card)
	if err != nil {
		return err
	}

	// special logic for already used yubikeys
	provisioned, err := s.isYubikeyProvisioned(card)
	if err != nil {
		return err
	}

	var conf YubiringConf
	if provisioned {
		conf, err = s.setupProvisionedYubikey(card)
	} else {
		conf, err = s.setupEmptyYubikey(card)
	}

	if err != nil {
		return err
	}

	if err := s.disableYubikeyOTP(card); err != nil {
		return fmt.Errorf("can't disable OTP interface: %w", err)
	}

	pinProvider, err := plain.NewProvider(conf.PIN)
	if err != nil {
		return fmt.Errorf("unable to create pin provider: %w", err)
	}

	yubi, err := yubiring.NewYubiring(card.Serial, pinProvider)
	if err != nil {
		return err
	}
	defer yubi.Close()

	return next(conf, yubi)
}

func (s *Setup) OpenYubikey(next func(YubiringConf, keyring.Keyring) error) error {
	s.LogInfo("initialize %s", yubiring.HumanName)

	checkCard := func(card yubikey.Card) error {
		yk, err := yubikey.Open(card)
		if err != nil {
			return fmt.Errorf("can't open yubikey %q: %w", card, err)
		}

		yk.Close()
		return nil
	}

	isYubikeyProvisioned := func(card yubikey.Card) (bool, error) {
		yk, err := yubikey.Open(card)
		if err != nil {
			return false, fmt.Errorf("can't open yubikey %q: %w", card, err)
		}

		_, err = yk.MngmtKey(yubikey.DefaultPIN)
		yk.Close()

		return err != nil && !errors.Is(err, yubikey.ErrNoManagementKey), nil
	}

	checkYubikeyAuth := func(yk *yubikey.Yubikey, pin string) error {
		_, err := yk.MngmtKey(pin)
		return err
	}

	cards, err := yubikey.Cards()
	if err != nil {
		return fmt.Errorf("failed to list yubikeys: %w", err)
	}

	card, err := asker.SelectYubikey(cards...)
	if err != nil {
		return err
	}

	err = checkCard(card)
	if err != nil {
		return err
	}

	provisioned, err := isYubikeyProvisioned(card)
	if err != nil {
		return err
	}

	if !provisioned {
		return errors.New("yubikey not provisioned, can't reuse them")
	}

	var pin string
	var message string
	for {
		pin, err = asker.AskYubiPIN(message)
		if err != nil {
			return err
		}

		yk, err := yubikey.Open(card)
		if err != nil {
			return fmt.Errorf("can't open yubikey %q: %w", card, err)
		}

		err = checkYubikeyAuth(yk, pin)
		if err != nil {
			message = fmt.Sprintf("Auth failed: %s", err)
		}

		yk.Close()
		if err == nil {
			break
		}
	}

	conf := YubiringConf{
		Serial: card.Serial,
		PIN:    pin,
	}

	pinProvider, err := plain.NewProvider(pin)
	if err != nil {
		return fmt.Errorf("unable to create pin provider: %w", err)
	}

	yubi, err := yubiring.NewYubiring(card.Serial, pinProvider)
	if err != nil {
		return err
	}
	defer yubi.Close()

	return next(conf, yubi)
}

func (s *Setup) NewKeychain(next func(KeychainConf, keyring.Keyring) error) error {
	s.LogInfo("initialize %s", keychain.HumanName)

	collections, err := keychain.Collections()
	if err != nil {
		return fmt.Errorf("failed to list secrets collections: %w", err)
	}

	collection, newOne, err := asker.SelectKeychainCollection(collections...)
	if err != nil {
		return err
	}

	if newOne {
		collection, err = keychain.CreateCollection(collection)
		if err != nil {
			return fmt.Errorf("failed to create new collection: %w", err)
		}
	}

	conf := KeychainConf{
		Collection: collection,
	}

	ring, err := keychain.NewKeychain(collection)
	if err != nil {
		return err
	}
	defer ring.Close()

	return next(conf, ring)
}

func (s *Setup) NewFilering(next func(FileringConf, keyring.Keyring) error) error {
	s.LogInfo("initialize %s", filering.HumanName)

	basePath, _ := paths.Filering()
	basePath, err := asker.AskFileringPath(basePath)
	if err != nil {
		return err
	}

	userPass, err := asker.ConfirmProvidePassphrase()
	if err != nil {
		return err
	}

	var passphrase string
	if userPass {
		passphrase, err = asker.AskNewPassphrase()
		if err != nil {
			return err
		}
	} else {
		passphrase, err = passutil.Password(32)
		if err != nil {
			return err
		}
	}

	conf := FileringConf{
		BasePath:   basePath,
		Passphrase: passphrase,
	}

	pinProvider, err := plain.NewProvider(passphrase)
	if err != nil {
		return fmt.Errorf("unable to create pin provider: %w", err)
	}

	ring, err := filering.NewFilering(conf.BasePath, pinProvider)
	if err != nil {
		return err
	}
	defer ring.Close()

	return next(conf, ring)
}

func (s *Setup) SelectSockets(next func([]config.Socket) error) error {
	s.LogInfo("initialize sockets")

	return selectSockets(next)
}

func (s *Setup) InstallService(next func(string) error) error {
	if ok, err := asker.ConfirmInstallService(); !ok {
		return err
	}

	startCmd, err := osutil.InstallService()
	if err != nil {
		return err
	}
	return next(startCmd)
}

func (s *Setup) UnInstallService(next func() error) error {
	err := osutil.UnInstallService()
	if err != nil {
		return err
	}
	return next()
}

func (s *Setup) WaitYubikey(card yubikey.Card) error {
	check := func() (bool, error) {
		cards, err := yubikey.Cards()
		if err != nil {
			return false, fmt.Errorf("can't listen yubikeys: %w", err)
		}

		for _, candidate := range cards {
			if candidate.Serial == card.Serial {
				return true, nil
			}
		}

		return false, nil
	}

	for {
		time.Sleep(500 * time.Millisecond)
		exists, err := check()
		if err != nil {
			return err
		}

		if exists {
			return nil
		}
	}
}
