package scenario

import (
	"errors"
	"fmt"

	"a.yandex-team.ru/security/skotty/skotty/internal/config"
	"a.yandex-team.ru/security/skotty/skotty/internal/keyring"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/asker"
	"a.yandex-team.ru/security/skotty/skotty/internal/socket"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil/sshclient"
)

var defaultSSHClients = []sshclient.ClientKind{
	sshclient.ClientKindWin32OpenSSH,
	sshclient.ClientPutty,
	sshclient.ClientKindCygwin,
	sshclient.ClientKindWSL,
}

func selectSockets(next func([]config.Socket) error) error {
	clients, err := asker.SelectSSHClients(defaultSSHClients...)
	if err != nil {
		return err
	}

	if len(clients) == 0 {
		return errors.New("no clients selected")
	}

	notify, err := asker.ConfirmSocketNotifications()
	if err != nil {
		return err
	}

	sockets := []config.Socket{
		{
			Name:        socket.NameDefault,
			NotifyUsage: notify,
			Kind:        socket.KindDummy,
			Path:        socket.NoPath,
			Keys: []keyring.KeyPurpose{
				keyring.KeyPurposeSecure,
				keyring.KeyPurposeInsecure,
				keyring.KeyPurposeSudo,
				keyring.KeyPurposeLegacy,
			},
		},
		{
			Name:        socket.NameSudo,
			NotifyUsage: notify,
			Kind:        socket.KindDummy,
			Path:        socket.NoPath,
			Keys: []keyring.KeyPurpose{
				keyring.KeyPurposeSudo,
			},
		},
	}

	for _, c := range clients {
		switch c {
		case sshclient.ClientKindWin32OpenSSH:
			sockets = append(sockets, config.Socket{
				Name:   c.SocketName(socket.NameDefault),
				Kind:   socket.KindPipe,
				Path:   `\\.\pipe\openssh-ssh-agent`,
				SameAs: socket.NameDefault,
			})
			sockets = append(sockets, config.Socket{
				Name:   c.SocketName(socket.NameSudo),
				Kind:   socket.KindPipe,
				Path:   `\\.\pipe\sudo-openssh-ssh-agent`,
				SameAs: socket.NameSudo,
			})
		case sshclient.ClientKindCygwin:
			sockets = append(sockets, config.Socket{
				Name:   c.SocketName(socket.NameDefault),
				Kind:   socket.KindCygwin,
				SameAs: socket.NameDefault,
			})
			sockets = append(sockets, config.Socket{
				Name:   c.SocketName(socket.NameSudo),
				Kind:   socket.KindCygwin,
				SameAs: socket.NameSudo,
			})
		case sshclient.ClientKindWSL:
			sockets = append(sockets, config.Socket{
				Name:   c.SocketName(socket.NameDefault),
				Kind:   socket.KindUnix,
				SameAs: socket.NameDefault,
			})
			sockets = append(sockets, config.Socket{
				Name:   c.SocketName(socket.NameSudo),
				Kind:   socket.KindUnix,
				SameAs: socket.NameSudo,
			})
		case sshclient.ClientPutty:
			sockets = append(sockets, config.Socket{
				Name:   c.SocketName(socket.NameDefault),
				Kind:   socket.KindPageant,
				Path:   socket.NoPath,
				SameAs: socket.NameDefault,
			})
		default:
			return fmt.Errorf("unsupported ssh client: %s", c)
		}
	}

	return next(sockets)
}
