package scenario

import (
	"errors"
	"fmt"

	"a.yandex-team.ru/security/skotty/skotty/internal/passutil"
	"a.yandex-team.ru/security/skotty/skotty/internal/setup/asker"
	"a.yandex-team.ru/security/skotty/skotty/internal/yubiconf"
	"a.yandex-team.ru/security/skotty/skotty/internal/yubikey"
)

func (s *Setup) tryOpenYubikey(card yubikey.Card) error {
	yk, err := yubikey.Open(card)
	if err != nil {
		return fmt.Errorf("can't open yubikey %q: %w", card, err)
	}

	yk.Close()
	return nil
}

func (s *Setup) isYubikeyProvisioned(card yubikey.Card) (bool, error) {
	yk, err := yubikey.Open(card)
	if err != nil {
		return false, fmt.Errorf("can't open yubikey %q: %w", card, err)
	}

	_, err = yk.MngmtKey(yubikey.DefaultPIN)
	yk.Close()

	return err != nil && !errors.Is(err, yubikey.ErrNoManagementKey), nil
}

func (s *Setup) checkYubikeyAuth(yk *yubikey.Yubikey, pin string) error {
	_, err := yk.MngmtKey(pin)
	if err != nil && errors.Is(err, yubikey.ErrNoManagementKey) {
		return nil
	}

	return err
}

func (s *Setup) resetYubikey(card yubikey.Card) (YubiringConf, error) {
	userPIN, err := asker.ConfirmProvidePIN()
	if err != nil {
		return YubiringConf{}, err
	}

	var pin, puk string
	if userPIN {
		pin, err = asker.AskNewPIN()
		if err != nil {
			return YubiringConf{}, err
		}

		puk, err = asker.AskNewPUK()
		if err != nil {
			return YubiringConf{}, err
		}
	} else {
		pin, err = passutil.Password(6)
		if err != nil {
			return YubiringConf{}, err
		}

		puk, err = passutil.Password(8)
		if err != nil {
			return YubiringConf{}, err
		}
	}

	yk, err := yubikey.Open(card)
	if err != nil {
		return YubiringConf{}, fmt.Errorf("can't open yubikey %q: %w", card, err)
	}

	err = yk.Reset(pin, puk)
	yk.Close()
	if err != nil {
		return YubiringConf{}, fmt.Errorf("can't reset yubikey %q: %w", card, err)
	}

	return YubiringConf{
		Serial:  card.Serial,
		PIN:     pin,
		PUK:     puk,
		userPIN: userPIN,
		userPUK: userPIN,
	}, nil
}

func (s *Setup) disableYubikeyOTP(card yubikey.Card) error {
	disable := func() (bool, error) {
		yc, err := yubiconf.Open(card)
		if err != nil {
			return false, fmt.Errorf("can't open yubikey %q: %w", card, err)
		}
		defer yc.Close()

		if !yc.IsUSBAppEnabled(yubiconf.ApplicationOTP) {
			return false, nil
		}

		s.LogInfo("disabling the OTP interface")

		if err := yc.DisableUSBApp(yubiconf.ApplicationOTP); err != nil {
			return false, err
		}

		if err := yc.WriteConfig(true); err != nil {
			return false, err
		}
		return true, nil
	}

	wait := func() error {
		s.LogWait("wait Yubikey reboot...")
		return s.WaitYubikey(card)
	}

	needWait, err := disable()
	if !needWait {
		return err
	}

	if err := wait(); err != nil {
		return err
	}

	s.LogSuccess("done")
	return nil
}

func (s *Setup) setupProvisionedYubikey(card yubikey.Card) (YubiringConf, error) {
	action, err := asker.ChooseYubiProvisionAction()
	if err != nil {
		return YubiringConf{}, err
	}

	checkDefaultPUK := func(yk *yubikey.Yubikey) (string, error) {
		if err := yk.SetPUK(yubikey.DefaultPUK, yubikey.DefaultPUK); err != nil {
			// that's fine, just not default PUK
			return "", nil
		}

		s.LogInfo("detected default PUK")
		userPUK, err := asker.ConfirmProvidePUK()
		if err != nil {
			return "", err
		}

		var puk string
		if userPUK {
			puk, err = asker.AskNewPUK()
			if err != nil {
				return "", err
			}
		} else {
			puk, err = passutil.Password(8)
			if err != nil {
				return "", err
			}
		}

		return puk, yk.SetPUK(yubikey.DefaultPUK, puk)
	}

	checkManagementKey := func(yk *yubikey.Yubikey, pin string) error {
		if yk.IsManagementKeyValid(&yubikey.DefaultManagementKey) {
			s.LogWarn("Yubikey uses default management key")
			s.LogInfo("generating new one")
			return yk.ResetMngmtKey(&yubikey.DefaultManagementKey)
		}

		_, err := yk.MngmtKey(pin)
		if err != nil {
			return fmt.Errorf("unable to get stored management key (probably you need to reset it first): %s", err)
		}

		return nil
	}

	if action == asker.YubiProvisionActionReset {
		return s.resetYubikey(card)
	}

	var pin, puk string
	var message string
	for {
		pin, err = asker.AskYubiPIN(message)
		if err != nil {
			return YubiringConf{}, err
		}

		yk, err := yubikey.Open(card)
		if err != nil {
			return YubiringConf{}, fmt.Errorf("can't open yubikey %q: %w", card, err)
		}

		err = s.checkYubikeyAuth(yk, pin)
		if err != nil {
			yk.Close()
			message = fmt.Sprintf("Auth failed: %s", err)
			continue
		}

		puk, err = checkDefaultPUK(yk)
		if err != nil {
			yk.Close()
			return YubiringConf{}, fmt.Errorf("unable to change default PUK: %w", err)
		}

		err = checkManagementKey(yk, pin)
		if err != nil {
			yk.Close()
			return YubiringConf{}, fmt.Errorf("unable to check management key: %w", err)
		}

		yk.Close()
		break
	}

	return YubiringConf{
		Serial:  card.Serial,
		PIN:     pin,
		PUK:     puk,
		userPIN: true,
	}, nil
}

func (s *Setup) setupEmptyYubikey(card yubikey.Card) (YubiringConf, error) {
	if ok, _ := asker.ConfirmReset(card); !ok {
		return YubiringConf{}, ErrAborted
	}

	return s.resetYubikey(card)
}
