package socket

import (
	"fmt"
	"net"
	"unsafe"

	"golang.org/x/sys/windows"
)

var (
	kernel32 = windows.NewLazySystemDLL("kernel32.dll")

	procGetNamedPipeClientProcessID    = kernel32.NewProc("GetNamedPipeClientProcessId")
	procGetNamedPipeClientProcessIDErr = procGetNamedPipeClientProcessID.Find()
)

func unixCreds(c net.Conn) (Creds, error) {
	winFile, ok := c.(interface{ Fd() uintptr })
	if !ok {
		return dummyCreds(), nil
	}

	var pid int32
	if err := getNamedPipeClientProcessID(windows.Handle(winFile.Fd()), &pid); err != nil {
		return dummyCreds(), fmt.Errorf("error in GetNamedPipeClientProcessId function: %w", err)
	}

	procName, procCmd, _ := procInfo(int(pid))
	return Creds{
		PID:  int(pid),
		Name: procName,
		Cmd:  procCmd,
	}, nil
}

func getNamedPipeClientProcessID(pipe windows.Handle, clientProcessID *int32) error {
	if procGetNamedPipeClientProcessIDErr != nil {
		return procGetNamedPipeClientProcessIDErr
	}

	r1, _, e1 := procGetNamedPipeClientProcessID.Call(uintptr(pipe), uintptr(unsafe.Pointer(clientProcessID)))
	if r1 == 0 {
		return e1
	}
	return nil
}
